/* $Id: AbstractActivityBehavior.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.ActivityBean;
import jp.co.argo21.nautica.workflow.dataaccess.ActivityDAO;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessBean;
import jp.co.argo21.nautica.workflow.dataaccess.VariableBean;
import jp.co.argo21.nautica.workflow.dataaccess.VariableDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.ApplicationDefinition;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.definition.ParticipantDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.definition.TransitionDefinition;
import jp.co.argo21.nautica.workflow.definition.VariableDefinition;
import jp.co.argo21.nautica.workflow.jms.WorkflowMessage;
import jp.co.argo21.nautica.workflow.util.DurationHelper;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.util.UniqueKeyGenerator;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;

/**
 * アクティビティの状態ごとの動作を定義するための抽象基底クラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public abstract class AbstractActivityBehavior implements ActivityBehavior
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/**
	 * 抽象アクティビティ動作を生成する。
	 */
	public AbstractActivityBehavior()
	{
	}

	/**
	 * 完了処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#complete(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal)
	 */
	public void complete(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act)
	throws Exception
	{
		ProcessBean pbean = proc.getBean();
		String pid = pbean.getID();
		ActivityBean abean = act.getBean();
		String bid = abean.getBlockID();

		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		VariableDAO varDAO = daoFactory.getVariableDAO();
		List<VariableBean> vars = varDAO.findByProcessIDAsList(pid);
		List<TransitionDefinition> trs = ConditionEvaluator.evaluate(pd, ad, vars, false);
		
		List<ActivityBean> acts = new ArrayList<ActivityBean>();
		for (TransitionDefinition tr : trs) {
			ActivityDefinition dest = tr.getDestination();
			
			//遷移先アクティビティの生成
			ActivityBean ab = createActivity(pd, dest, pbean, bid);
			acts.add(ab);
		}
		
		if (acts.size() == 0) {
			//遷移先が見つからない。定義ミスの可能性あり。
			suspend(pd, ad, proc, act);
			String F0001 = StringManager.get("F0001");
			eLog.fatal(F0001);
			return;
		}
		
		//該当アクティビティを完了させる。
		//アクティビティの状態をCLOSED_COMPLETEDの状態にする。
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.CLOSED_COMPLETED, ts);
		
			
		//遷移先アクティビティをメッセージキューに登録する。
		for (ActivityBean ab : acts) {
			String aid = ab.getActivityID();
			WorkflowMessage sarmsg = new StartActivityRequestMessage(pid, aid);
			SystemMessageTransporter mt = new SystemMessageTransporter(StartActivityRequestWatcher.NAME);
			mt.sendMessage(sarmsg);
		}
	}


	/**
	 * 保留処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#suspend(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal)
	 */
	public void suspend(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act)
	throws Exception
	{
		//該当アクティビティを一時停止させる。
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.OPEN_NOT_RUNNING_SUSPENDED, ts);
	}

	/**
	 * 復帰処理を行う。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param proc プロセス
	 * @param act アクティビティ
	 * @throws Exception 任意の例外
	 * @see jp.co.argo21.nautica.workflow.engine.ActivityBehavior#resume(jp.co.argo21.nautica.workflow.definition.ProcessDefinition, jp.co.argo21.nautica.workflow.definition.ActivityDefinition, jp.co.argo21.nautica.workflow.engine.ProcessInternal, jp.co.argo21.nautica.workflow.engine.ActivityInternal)
	 */
	public void resume(ProcessDefinition pd, ActivityDefinition ad,
			ProcessInternal proc, ActivityInternal act)
	throws Exception
	{
		//該当アクティビティを復帰させる。
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		Timestamp ts = DataAccessManager.getAccessTime();
		actDAO.updateState(act.getBean(), ActivityState.OPEN_RUNNING, ts);
	}

	/**
	 * 引数に指定された情報から新規アクティビティインスタンスを生成する。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param pbean プロセス
	 * @param bid ブロックID
	 * @return 新規に作られたアクティビティ
	 * @throws Exception
	 */
	protected ActivityBean createActivity(
			ProcessDefinition pd, ActivityDefinition ad, ProcessBean pbean, String bid)
	throws Exception
	{
		//レイヤー内のアクティビティの場合
		if (! (bid == null || bid.trim().equals(""))) {
			String type = ad.getType();
			if (type.equals(ActivityDefinition.CONCUR_END)) {
				ActivityBean ceabean = getConcurrentEndActivity(pd, ad, pbean, bid);
				//すでに別な遷移によって該当同期終了アクティビティが作られている場合
				//それを返す。
				if (ceabean != null) {
					return ceabean;
				}
			}
		}
		
		Timestamp ts = DataAccessManager.getAccessTime();
		int du = pd.getDurationUnit();
		int limit = ad.getLimit();
		long durationLimit = ts.getTime() + DurationHelper.calculateDuration(du, limit);
		Timestamp limitTs = new Timestamp(durationLimit);
		
		ActivityBean abean = new ActivityBean();
		abean.setActivityDefinitionID(ad.getID());
		abean.setActivityID(UniqueKeyGenerator.generate(ActivityBean.ID_PREFIX));
		abean.setActivityState(ActivityState.OPEN_NOT_RUNNING_NOT_STARTED.toInt());
		abean.setActorID(ParticipantImpl.SYSTEM_ID);
		abean.setDurationLimit(limitTs);
		abean.setProcessID(pbean.getID());
		abean.setStartDate(ts);
		abean.setStarterID(pbean.getStarterID());
		abean.setUpdateDate(ts);
		abean.setBlockID(bid);
		
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		actDAO.insert(abean);
		
		return abean;
	}
	
	/**
	 * 引数情報で特定されるワークフロー参加者定義を返す。
	 *
	 * @param pdid プロセス定義ID
	 * @param performerID ワークフロー参加者定義ID
	 * @return ワークフロー参加者定義
	 * @throws Exception 任意の例外
	 */
	protected ParticipantDefinition getParticipantDefinition(
			String pdid, String performerID)
	throws Exception
	{
		WorkflowEngine engine = WorkflowEngine.getInstance();
		DefinitionRepositoryImpl dr = engine.getDefinitionRepository();
		PackageDefinition pkgd = dr.getPackageDefinition(pdid);
		List<ParticipantDefinition> parts = pkgd.getParticipants();
		ParticipantDefinition partd = null;
		for (ParticipantDefinition def : parts) {
			if (def.getID().equals(performerID)) {
				partd = def;
				break;
			}
		}
		return partd;
	}
	
	/**
	 * 引数情報で特定されるアプリケーション定義を返す。
	 *
	 * @param pdid プロセス定義ID
	 * @param appID アプリケーション定義ID
	 * @return アプリケーション定義
	 * @throws Exception 任意の例外
	 */
	protected ApplicationDefinition getApplicationDefinition(
			String pdid, String appID)
	throws Exception
	{
		WorkflowEngine engine = WorkflowEngine.getInstance();
		DefinitionRepositoryImpl dr = engine.getDefinitionRepository();
		PackageDefinition pkgd = dr.getPackageDefinition(pdid);
		List<ApplicationDefinition> apps = pkgd.getApplications();
		ApplicationDefinition appd = null;
		for (ApplicationDefinition def : apps) {
			if (def.getID().equals(appID)) {
				appd = def;
				break;
			}
		}
		return appd;
	}
	
	/**
	 * 引数情報で特定されるプロセス変数定義を返す。
	 *
	 * @param pdid プロセス定義ID
	 * @param varID プロセス変数定義ID
	 * @return プロセス変数定義
	 * @throws Exception 任意の例外
	 */
	protected VariableDefinition getVariableDefinition(
			String pdid, String varID)
	throws Exception
	{
		WorkflowEngine engine = WorkflowEngine.getInstance();
		DefinitionRepositoryImpl dr = engine.getDefinitionRepository();
		PackageDefinition pkgd = dr.getPackageDefinition(pdid);
		List<VariableDefinition> vars = pkgd.getVariables();
		VariableDefinition vard = null;
		for (VariableDefinition def : vars) {
			if (def.getID().equals(varID)) {
				vard = def;
				break;
			}
		}
		return vard;
	}

	/**
	 * 引数に指定された情報から同期終了アクティビティインスタンスを検索する。
	 *
	 * @param pd プロセス定義
	 * @param ad アクティビティ定義
	 * @param pbean プロセス
	 * @param bid ブロックID
	 * @return 新規に作られたアクティビティ
	 * @throws Exception
	 */
	private ActivityBean getConcurrentEndActivity(
			ProcessDefinition pd, ActivityDefinition ad, ProcessBean pbean, String bid)
	throws Exception
	{
		String pid = pbean.getID();
		String adid = ad.getID();
		WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
		ActivityDAO actDAO = daoFactory.getActivityDAO();
		ActivityBean abean = actDAO.findConcurrentEnd(pid, adid, bid);
		
		return abean;
	}
}
