/* $Id: WorkflowServiceManagerBean.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ejb;


import javax.ejb.EJB;
import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import jp.co.argo21.nautica.workflow.wfmc.ConnectionFailedException;
import jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowAdminHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowServiceManager;
/**
 * WorkflowServiceManagerBeanは、ローカルのWorkflowServiceManagerに対する
 * リモート呼び出しを実現する具体実装クラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowServiceManager
 */
@Stateless
@Remote(WorkflowServiceManager.class)
@TransactionManagement(TransactionManagementType.CONTAINER)
public class WorkflowServiceManagerBean
implements WorkflowServiceManager
{
	@EJB
	private DefinitionRepository definitionRepository;
	@EJB
	private WorkflowEngineHandler workflowEngineHandler;
	@EJB
	private WorkflowAdminHandler workflowAdminHandler;
	@EJB
	private WorkItemHandler workItemHandler;

	/**
	 * ワークフローエンジンへの接続認証を行い、
	 * セッションIDを取得する。
	 *
	 * @param user ユーザID
	 * @param pass パスワード
	 * @return セッションID
	 * @throws ConnectionFailedException 接続認証に失敗した場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowServiceManagerRemote#connect(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public String connect(String user, String pass)
	throws ConnectionFailedException
	{
		WorkflowServiceManager manager = ComponentUtils.getWorkflowServiceManager();
		return manager.connect(user, pass);
	}

	/**
	 * ワークフローエンジンとの切断を行う。
	 *
	 * @param session セッションID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowServiceManagerRemote#disconnect(java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void disconnect(String session)
	throws InvalidSessionException
	{
		WorkflowServiceManager manager = ComponentUtils.getWorkflowServiceManager(session);
		manager.disconnect(session);
	}

	/**
	 * 定義リポジトリを返す。
	 *
	 * @param session セッションID
	 * @return 定義リポジトリ
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowServiceManagerRemote#getDefinitionRepository(java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public DefinitionRepository getDefinitionRepository(String session)
	throws InvalidSessionException
	{
		ComponentUtils.getDefinitionRepository(session);
		return definitionRepository;
	}

	/**
	 * 作業項目ハンドラを返す。
	 *
	 * @param session セッションID
	 * @return 作業項目ハンドラ
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowServiceManagerRemote#getWorkItemHandler(java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public WorkItemHandler getWorkItemHandler(String session)
	throws InvalidSessionException
	{
		ComponentUtils.getWorkItemHandler(session);
		return workItemHandler;
	}

	/**
	 * ワークフローエンジンハンドラを返す。
	 *
	 * @param session セッションID
	 * @return ワークフローエンジンハンドラ
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowServiceManagerRemote#getWorkflowEngineHandler(java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public WorkflowEngineHandler getWorkflowEngineHandler(String session)
	throws InvalidSessionException
	{
		ComponentUtils.getWorkflowEngineHandler(session);
		return workflowEngineHandler;
	}

	/**
	 * ワークフロー管理ハンドラを返す。
	 *
	 * @param session セッションID
	 * @return ワークフロー管理ハンドラ
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowServiceManagerRemote#getWorkflowAdminHandler(java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public WorkflowAdminHandler getWorkflowAdminHandler(String session)
	throws InvalidSessionException
	{
		ComponentUtils.getWorkflowAdminHandler(session);
		return workflowAdminHandler;
	}
}
