/* $Id: WorkflowEngineHandlerBean.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ejb;


import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import jp.co.argo21.nautica.workflow.wfmc.Activity;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.AttributeAssignmentFailedException;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidActivityInstanceException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidAttributeException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessDefinitionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessInstanceException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.Process;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler;

/**
 * WorkflowEngineHandlerBeanは、ローカルのWorkflowEngineHandlerに対する
 * リモート呼び出しを実現する具体実装クラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler
 */
@Stateless
@Remote(WorkflowEngineHandler.class)
@TransactionManagement(TransactionManagementType.CONTAINER)
public class WorkflowEngineHandlerBean
implements WorkflowEngineHandler
{
	/**
	 * プロセスインスタンスの生成を行う。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param name プロセスインスタンス名称
	 * @return プロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#createProcess(java.lang.String, java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public String createProcess(String session, String pdid, String name)
	throws InvalidSessionException, InvalidProcessDefinitionException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.createProcess(session, pdid, name);
	}

	/**
	 * プロセスインスタンスの開始を行う。
	 * 開始した場合に、プロセスインスタンスIDを更新するような
	 * 実装も可能である。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @return 更新されたプロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#startProcess(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public String startProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidAttributeException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.startProcess(session, pid);
	}

	/**
	 * プロセスインスタンスの停止を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#terminateProcess(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void terminateProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		engineHandler.terminateProcess(session, pid);
	}

	/**
	 * 指定されたプロセスインスタンスが利用可能な状態一覧の取得を行う。
	 * 利用可能なフィルターはStateFilterのみとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param filter 検索用フィルター
	 * @return プロセスインスタンス状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getProcessStates(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public ProcessState[] getProcessStates(String session, 
			String pid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidFilterException
	{	
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getProcessStates(session, pid, filter);
	}


	/**
	 * プロセスインスタンス状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param state プロセスインスタンス状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#changeProcessState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.ProcessState)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void changeProcessState(String session,
			String pid, ProcessState state)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidStateException, TransitionNotAllowedException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		engineHandler.changeProcessState(session, pid, state);
	}

	/**
	 * プロセスインスタンス属性一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, AttributeFilter, GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param filter 検索用フィルター
	 * @return 属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getProcessAttributes(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Attribute[] getProcessAttributes(String session,
			String pid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidFilterException
	{		
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getProcessAttributes(session, pid, filter);
	}

	/**
	 * プロセスインスタンス属性の属性値の取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getProcessAttributeValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Attribute getProcessAttributeValue(String session,
			String pid, String name)
	throws InvalidSessionException, InvalidAttributeException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getProcessAttributeValue(session, pid, name);
	}

	/**
	 * プロセスインスタンス属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#assignProcessAttribute(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void assignProcessAttribute(String session,
			String pid, Attribute attr)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidAttributeException, AttributeAssignmentFailedException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		engineHandler.assignProcessAttribute(session, pid, attr);
	}

	/**
	 * 指定されたアクティビティインスタンスが利用可能な状態一覧の取得を行う。
	 * 利用可能なフィルターはStateFilterのみとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getActivityStates(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public ActivityState[] getActivityStates(String session,
			String pid, String aid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidFilterException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getActivityStates(session, pid, aid, filter);
	}

	/**
	 * アクティビティインスタンス状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param state アクティビティインスタンス状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#changeActivityState(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.ActivityState)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void changeActivityState(String session,
			String pid, String aid, ActivityState state)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidStateException,
		TransitionNotAllowedException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		engineHandler.changeActivityState(session, pid, aid, state);
	}

	/**
	 * アクティビティインスタンス属性一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, AttributeFilter, GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getActivityAttributes(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Attribute[] getActivityAttributes(String session,
			String pid, String aid, Filter filter)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidFilterException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getActivityAttributes(session, pid, aid, filter);
	}

	/**
	 * アクティビティインスタンス属性の属性値の取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getActivityAttributeValue(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Attribute getActivityAttributeValue(String session,
			String pid, String aid, String name)
	throws InvalidSessionException, InvalidAttributeException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getActivityAttributeValue(session, pid, aid, name);
	}

	/**
	 * アクティビティインスタンス属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#assignActivityAttribute(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void assignActivityAttribute(String session,
			String pid, String aid, Attribute attr)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException, InvalidAttributeException,
		AttributeAssignmentFailedException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		engineHandler.assignActivityAttribute(session, pid, aid, attr);
	}

	/**
	 * プロセスインスタンス一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, StateFilter、GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return プロセスの一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getProcesses(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Process[] getProcesses(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getProcesses(session, filter);
	}

	/**
	 * プロセスインスタンスの取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @return プロセスインスタンス
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getProcess(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Process getProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getProcess(session, pid);
	}

	/**
	 * アクティビティインスタンス一覧の取得を行う。
	 * 利用可能なフィルターはAttributeFilter, StateFilter、GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getActivities(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Activity[] getActivities(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getActivities(session, filter);
	}

	/**
	 * アクティビティインスタンス一覧の取得を行う。
	 * 利用可能なフィルターはAttributeFilter, StateFilter、GroupFilterとする。
	 * WfMC仕様ではpidはないが、これは誤りであろうと判断し、
	 * pidを引数に追加した。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param filter 検索用フィルター
	 * @return アクティビティインスタンス一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getActivities(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Activity[] getActivities(String session, String pid, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getActivities(session, pid, filter);
	}

	/**
	 * アクティビティインスタンスの取得を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @return アクティビティインスタンス
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @throws InvalidActivityInstanceException アクティビティインスタンスIDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowEngineHandlerRemote#getActivity(java.lang.String, java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Activity getActivity(String session, String pid, String aid)
	throws InvalidSessionException, InvalidProcessInstanceException,
		InvalidActivityInstanceException
	{
		WorkflowEngineHandler engineHandler = ComponentUtils.getWorkflowEngineHandler(session);
		return engineHandler.getActivity(session, pid, aid);
	}
}
