/* $Id: WorkflowAdminHandlerBean.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ejb;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidActivityNameException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidAttributeException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessDefinitionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessInstanceException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowAdminHandler;

/**
 * WorkflowAdminHandlerBeanは、ローカルのWorkflowAdminHandlerに対する
 * リモート呼び出しを実現する具体実装クラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 * @see jp.co.argo21.nautica.workflow.wfmc.WorkflowAdminHandler
 */
@Stateless
@Remote(WorkflowAdminHandler.class)
@TransactionManagement(TransactionManagementType.CONTAINER)
public class WorkflowAdminHandlerBean
implements WorkflowAdminHandler
{
	/**
	 * 指定されたプロセス定義に属するプロセスインスタンスを
	 * フィルターで絞り込み、状態の変更を行う。
	 * 指定可能なフィルターはStateFilterとする。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param filter 検索用フィルター
	 * @param state 新しいプロセスインスタンス状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#changeProcessesState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter, jp.co.argo21.nautica.workflow.wfmc.ProcessState)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void changeProcessesState(String session,
			String pdid, Filter filter, ProcessState state)
	throws InvalidSessionException, InvalidProcessDefinitionException,
	InvalidFilterException, InvalidStateException,
	TransitionNotAllowedException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.changeProcessesState(session, pdid, filter, state);
	}

	/**
	 * 指定されたプロセス定義とアクティビティ定義に属する
	 * アクティビティインスタンスをフィルターで絞り込み、状態の変更を行う。
	 * 指定可能なフィルターはStateFilterとする。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param adid アクティビティ定義ID
	 * @param filter 検索用フィルター
	 * @param state 新しいアクティビティインスタンス状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidActivityNameException 指定されたアクティビティ定義IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#changeActivitiesState(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter, jp.co.argo21.nautica.workflow.wfmc.ActivityState)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void changeActivitiesState(String session,
			String pdid, String adid, Filter filter, ActivityState state)
	throws InvalidSessionException, InvalidProcessDefinitionException,
	InvalidActivityNameException, InvalidFilterException,
	InvalidStateException, TransitionNotAllowedException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.changeActivitiesState(session, pdid, adid, filter, state);
	}

	/**
	 * 指定されたプロセス定義に属するプロセスインスタンスを
	 * フィルターで絞り込み、停止を行う。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param filter 検索用フィルター
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#terminateProcesses(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void terminateProcesses(String session,
			String pdid, Filter filter)
	throws InvalidSessionException, InvalidProcessDefinitionException,
		InvalidFilterException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.terminateProcesses(session, pdid, filter);
	}

	/**
	 * 指定されたプロセス定義に属するプロセスインスタンスを
	 * フィルターで絞り込み、属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param filter 検索用フィルター
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#assignProcessesAttribute(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void assignProcessesAttribute(String session,
			String pdid, Filter filter, Attribute attr)
	throws InvalidSessionException, InvalidProcessDefinitionException,
		InvalidFilterException, InvalidAttributeException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.assignProcessesAttribute(session, pdid, filter, attr);
	}

	/**
	 * 指定されたプロセス定義とアクティビティ定義に属する
	 * アクティビティインスタンスをフィルターで絞り込み、
	 * 属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param adid アクティビティ定義ID
	 * @param filter 検索用フィルター
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidActivityNameException 指定されたアクティビティ定義IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#assignActivitiesAttribute(java.lang.String, java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void assignActivitiesAttribute(String session,
			String pdid, String adid, Filter filter, Attribute attr)
	throws InvalidSessionException, InvalidProcessDefinitionException,
		InvalidActivityNameException, InvalidFilterException,
		InvalidAttributeException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.assignActivitiesAttribute(session, pdid, adid, filter, attr);
	}

	/**
	 * 指定されたプロセス定義に属するプロセスインスタンスを
	 * フィルターで絞り込み、中断を行う。
	 *
	 * @param session セッションID
	 * @param pdid プロセス定義ID
	 * @param filter 検索用フィルター
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessDefinitionException 指定されたプロセス定義IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#abortProcesses(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void abortProcesses(String session,
			String pdid, Filter filter)
	throws InvalidSessionException, InvalidProcessDefinitionException,
		InvalidFilterException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.abortProcesses(session, pdid, filter);
	}

	/**
	 * 指定されたプロセスインスタンスの中断を行う。
	 *
	 * @param session セッションID
	 * @param pid プロセスインスタンスID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidProcessInstanceException プロセスインスタンスIDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#abortProcess(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void abortProcess(String session, String pid)
	throws InvalidSessionException, InvalidProcessInstanceException
	{
		WorkflowAdminHandler adminHandler = ComponentUtils.getWorkflowAdminHandler(session);
		adminHandler.abortProcess(session, pid);
	}

	/**
	 * 指定された定義の登録処理を行い、エンジンに認識させる。
	 *
	 * @param session セッションID
	 * @param name ワークフロー定義名
	 * @param path ワークフロー定義の相対パス
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws WorkflowException プロセスインスタンスIDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#registerDefinition(java.lang.String, java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void registerDefinition(String session, String name, String path)
	throws InvalidSessionException, WorkflowException
	{
		DefinitionRepository repository = ComponentUtils.getDefinitionRepository(session);
		repository.registerDefinition(session, name, path);
	}

	/**
	 * ワークフローエンジンのシャットダウンを行う。
	 *
	 * @param session セッションID
	 * @param adminKey 管理キー
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws WorkflowException シャットダウンに失敗した場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkflowAdminHandlerRemote#shutdown(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void shutdown(String session, String adminKey)
	throws InvalidSessionException, WorkflowException
	{
		//TODO : NOP
	}
}
