/* $Id: RoleManagerBean.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ejb;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import jp.co.argo21.nautica.workflow.security.InvalidOrganizationException;
import jp.co.argo21.nautica.workflow.security.InvalidRoleException;
import jp.co.argo21.nautica.workflow.security.InvalidUserException;
import jp.co.argo21.nautica.workflow.security.Organization;
import jp.co.argo21.nautica.workflow.security.Role;
import jp.co.argo21.nautica.workflow.security.RoleManager;
import jp.co.argo21.nautica.workflow.security.User;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowSecurityException;

/**
 * RoleManagerBean は、ローカルの RoleManagerRemote に対する
 * リモート呼び出しを実現する具体実装クラスである。
 *
 * @author  ysahara(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
@Stateless
@Remote(RoleManager.class)
@TransactionManagement(TransactionManagementType.CONTAINER)
public class RoleManagerBean implements RoleManager
{
	/**
	 * ridに対応する権限を返す。
	 * ridに対応する権限がない場合は null を返す。
	 *
	 * @param rid 権限ID
	 * @return 権限
	 * @throws InvalidRoleException 指定された権限が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Role getRole(String rid)
	throws InvalidRoleException, WorkflowSecurityException
	{
		RoleManager rm = ComponentUtils.getRoleManager();
		return rm.getRole(rid);
	}

	/**
	 * 組織に付与されている全権限を返す。
	 * 権限が存在しない場合は空の配列を返す。
	 *
	 * @param org 組織単位
	 * @return 権限の一覧
	 * @throws InvalidOrganizationException
	 * @throws WorkflowSecurityException
	 * @throws InvalidRoleException 指定された組織が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Role[] getAttachedRoles(Organization org)
	throws InvalidOrganizationException, WorkflowSecurityException
	{
		RoleManager rm = ComponentUtils.getRoleManager();
		return rm.getAttachedRoles(org);
	}

	/**
	 * ユーザに付与されている全権限を返す。
	 * 権限が存在しない場合は空の配列を返す。
	 *
	 * @param user ユーザ
	 * @return 権限の一覧
	 * @throws InvalidRoleException 指定されたユーザが有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Role[] getAttachedRoles(User user)
	throws InvalidUserException, WorkflowSecurityException
	{
		RoleManager rm = ComponentUtils.getRoleManager();
		return rm.getAttachedRoles(user);
	}

	/**
	 * 権限を持つ全組織単位を返す。
	 * 組織が存在しない場合は空の配列を返す。
	 *
	 * @param role 権限
	 * @return 組織単位の一覧
	 * @throws InvalidRoleException 指定された権限が有効でない場合
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Organization[] getOrganizationsByRole(Role role)
	throws InvalidRoleException, WorkflowSecurityException
	{
		RoleManager rm = ComponentUtils.getRoleManager();
		return rm.getOrganizationsByRole(role);
	}

	/**
	 * 権限を持つ全ユーザIDを返す。
	 * ユーザが存在しない場合は空の配列を返す。
	 *
	 * @param role 権限
	 * @return ユーザの一覧
	 * @throws InvalidRoleException 指定された権限が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public User[] getUsersByRole(Role role)
	throws InvalidRoleException, WorkflowSecurityException
	{
		RoleManager rm = ComponentUtils.getRoleManager();
		return rm.getUsersByRole(role);
	}
}
