/* $Id: TransitionDefinitionImpl.java,v 1.3 2007/11/20 08:29:38 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2002.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.definition.impl;

import java.util.List;

import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.TransitionDefinition;
import jp.co.argo21.nautica.workflow.expression.ExceptionExpression;
import jp.co.argo21.nautica.workflow.expression.Expression;
import jp.co.argo21.nautica.workflow.expression.LogicalExpression;
import jp.co.argo21.nautica.workflow.expression.OtherwiseExpression;
import jp.co.argo21.nautica.workflow.expression.RelationalExpression;
import jp.co.argo21.nautica.workflow.expression.UnaryExpression;
import jp.co.argo21.nautica.workflow.util.IDGenerator;
import jp.co.argo21.nautica.workflow.util.StringUtils;
import jp.co.argo21.nautica.workflow.xpdl.ConditionalExpressionDocument;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlObject;
import org.wfmc.x2002.xpdl10.ConditionDocument;
import org.wfmc.x2002.xpdl10.TransitionDocument;

/**
 * TransitionDefinitionインタフェースの実装クラスである。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class TransitionDefinitionImpl implements TransitionDefinition
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 遷移元アクティビティ */
	private ActivityDefinition source;
	/** 遷移先アクティビティ */
	private ActivityDefinition destination;
	/** 遷移条件 */
	private Expression condition;
	/** 遷移型 */
	private int type;

	/**
	 * 遷移を生成する。
	 * 
	 * @param tr	XmlObject
	 * @param acts	アクティビティ一覧
	 */
	public TransitionDefinitionImpl(TransitionDocument.Transition tr, List<ActivityDefinition> acts)
	{
		String id = tr.getId();
		if (StringUtils.checkNull(id)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(tr.getName());
		setDescription(tr.getDescription());
		
		String sid = tr.getFrom();
		setSource(getActivityDefinition(acts, sid));
		
		String did = tr.getTo();
		setDestination(getActivityDefinition(acts, did));
		
		if (tr.isSetCondition()) {
			ConditionDocument.Condition c = tr.getCondition();
			int type = c.getType().intValue();
			try {
				if (type == CONDITION) {
					XmlObject obj = XmlObject.Factory.parse(c.toString());
					//値が設定されている場合のみConditionalExpressionDocument型の
					//オブジェクトが返されてくる。
					if (obj == null) {
						setCondition(null);
					} else {
						ConditionalExpressionDocument ceDoc
							= (ConditionalExpressionDocument)obj.changeType(ConditionalExpressionDocument.type);
						ConditionalExpressionDocument.ConditionalExpression ce
							= ceDoc.getConditionalExpression();
						if (ce.isSetRelationalExpression()) {
							setCondition(new RelationalExpression(ce.getRelationalExpression()));
						} else if (ce.isSetUnaryExpression()) {
							setCondition(new UnaryExpression(ce.getUnaryExpression()));
						} else if (ce.isSetLogicalExpression()) {
							setCondition(new LogicalExpression(ce.getLogicalExpression()));
						}
					}
				} else if (type == OTHERWISE) {
					setCondition(new OtherwiseExpression());
				} else if (type == EXCEPTION) {
					setCondition(new ExceptionExpression());
				}
			} catch (XmlException ex) {
				//NOP
			}
		} else {
			setCondition(null);
		}
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * 名称を返す。
	 * 
	 * @return 名称
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * 遷移条件を返す。
	 * 
	 * @return	遷移条件
	 * @see jp.co.argo21.nautica.workflow.definition.TransitionDefinition#getCondition()
	 */
	public Expression getCondition()
	{
		return condition;
	}

	/**
	 * 遷移先アクティビティを返す。
	 * 
	 * @return	遷移先アクティビティ
	 * @see jp.co.argo21.nautica.workflow.definition.TransitionDefinition#getDestination()
	 */
	public ActivityDefinition getDestination()
	{
		return destination;
	}

	/**
	 * 遷移元アクティビティを返す。
	 * 
	 * @return	遷移元アクティビティ
	 * @see jp.co.argo21.nautica.workflow.definition.TransitionDefinition#getSource()
	 */
	public ActivityDefinition getSource()
	{
		return source;
	}

	/**
	 * 遷移型を返す。
	 * 
	 * @return	遷移型
	 * @see jp.co.argo21.nautica.workflow.definition.TransitionDefinition#getType()
	 */
	public int getType()
	{
		return type;
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * 遷移元アクティビティを設定する。
	 * 
	 * @param act	遷移元アクティビティ
	 */
	private void setSource(ActivityDefinition act)
	{
		this.source = act;
	}

	/**
	 * 遷移先アクティビティを設定する。
	 * 
	 * @param act	遷移先アクティビティ
	 */
	private void setDestination(ActivityDefinition act)
	{
		this.destination = act;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 */
	private void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * 名称を設定する。
	 * 
	 * @param name 名称
	 */
	private void setName(String name)
	{
		this.name = name;
	}

	/**
	 * 遷移条件を設定する。
	 * 
	 * @param cond	遷移条件
	 */
	private void setCondition(Expression cond)
	{
		if (cond == null) {
			type = CONDITION;
		} else if (cond instanceof OtherwiseExpression) {
			type = OTHERWISE;
		} else if (cond instanceof ExceptionExpression) {
			type = EXCEPTION;
		} else {
			type = CONDITION;
		}
		condition = cond;
	}

	/**
	 * 遷移型を設定する。
	 * 
	 * @param i	遷移型
	 */
	@SuppressWarnings("unused")
	private void setType(int i)
	{
		type = i;
	}

	/**
	 * 指定されたIDを持つアクティビティを返す。
	 * 
	 * @param acts	アクティビティ一覧
	 * @param id	アクティビティID
	 * @return	アクティビティ
	 */
	private ActivityDefinition getActivityDefinition(List<ActivityDefinition> acts, String id)
	{
		for (ActivityDefinition act : acts) {
			String aid = act.getID();
			if (id.equals(aid)) return act;
		}
		return null;
	}
}
