/* $Id: ProcessDefinitionImpl.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.definition.impl;

import java.util.ArrayList;
import java.util.List;

import org.wfmc.x2002.xpdl10.ActivitiesDocument;
import org.wfmc.x2002.xpdl10.ActivityDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributeDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributesDocument;
import org.wfmc.x2002.xpdl10.FormalParameterDocument;
import org.wfmc.x2002.xpdl10.FormalParametersDocument;
import org.wfmc.x2002.xpdl10.ProcessHeaderDocument;
import org.wfmc.x2002.xpdl10.RedefinableHeaderDocument;
import org.wfmc.x2002.xpdl10.ResponsiblesDocument;
import org.wfmc.x2002.xpdl10.TransitionDocument;
import org.wfmc.x2002.xpdl10.TransitionsDocument;
import org.wfmc.x2002.xpdl10.WorkflowProcessDocument;

import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.ParameterDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.definition.TransitionDefinition;
import jp.co.argo21.nautica.workflow.util.IDGenerator;
import jp.co.argo21.nautica.workflow.util.StringUtils;

/**
 * ProcessDefinitionインタフェースの実装クラスである。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessDefinitionImpl implements ProcessDefinition
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** 未初期化時の仮ID */
	private static final String UNINITIALIZED_ID = "--UNINITIALIZED--"; 

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 期間単位 */
	private int durationUnit;
	/** 実行期限 */
	private int limit;
	/** アクセスレベル */
	private int accessLevel;
	/** 論理座標幅 */
	private int logicalWidth;
	/** 論理座標高さ */
	private int logicalHeight;
	/** 通知者一覧 */
	private List<String> responsibles = new ArrayList<String>();
	/** パラメータ一覧 */
	private List<ParameterDefinition> parameters = new ArrayList<ParameterDefinition>();
	/** アクティビティ一覧 */
	private List<ActivityDefinition> activities = new ArrayList<ActivityDefinition>();
	/** 遷移一覧 */
	private List<TransitionDefinition> transitions = new ArrayList<TransitionDefinition>();

	/**
	 * プロセスを生成する。
	 * 
	 * @param proc	XmlObject
	 */
	public ProcessDefinitionImpl(WorkflowProcessDocument.WorkflowProcess proc)
	{
		String id = proc.getId();
		if (StringUtils.checkNull(id) || id.equals(UNINITIALIZED_ID)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(proc.getName());
		
		ProcessHeaderDocument.ProcessHeader ph = proc.getProcessHeader();
		setDescription(ph.getDescription());
		setDurationUnit(ph.getDurationUnit().intValue());
		try {
			setLimit(Integer.parseInt(ph.getLimit()));
		} catch (Exception ex) {
			setLimit(0);
		}
		setAccessLevel(proc.getAccessLevel().intValue());

		if (proc.isSetRedefinableHeader()) {
			RedefinableHeaderDocument.RedefinableHeader rhd
				= proc.getRedefinableHeader();
			if (rhd.isSetResponsibles()) {
				ResponsiblesDocument.Responsibles rsps = rhd.getResponsibles();
				int count = rsps.sizeOfResponsibleArray();
				for (int i = 0; i < count; i++) {
					String rsp = rsps.getResponsibleArray(i);
					addResponsible(rsp);
				}
			}
		}

		if (proc.isSetExtendedAttributes()) {
			ExtendedAttributesDocument.ExtendedAttributes eas = proc.getExtendedAttributes();
			int count = eas.sizeOfExtendedAttributeArray();
			for (int i = 0; i < count; i++) {
				ExtendedAttributeDocument.ExtendedAttribute ea = eas.getExtendedAttributeArray(i);
				String n = ea.getName();
				String v = ea.getValue();
				if (n.equals(LOGICAL_WIDTH)) {
					setLogicalWidth(Integer.parseInt(v));
				} else if (n.equals(LOGICAL_HEIGHT)) {
					setLogicalHeight(Integer.parseInt(v));
				}
			}
		}
		
		if (proc.isSetFormalParameters()) {
			FormalParametersDocument.FormalParameters pms = proc.getFormalParameters();
			int count = pms.sizeOfFormalParameterArray();
			for (int i = 0; i < count; i++) {
				FormalParameterDocument.FormalParameter pm = pms.getFormalParameterArray(i);
				ParameterDefinition e = new ParameterDefinitionImpl(pm);
				addParameter(e);
			}
		}
		
		if (proc.isSetActivities()) {
			ActivitiesDocument.Activities acts = proc.getActivities();
			int count = acts.sizeOfActivityArray();
			for (int i = 0; i < count; i++) {
				ActivityDocument.Activity act = acts.getActivityArray(i);
				ActivityDefinition e = new ActivityDefinitionImpl(act);
				addActivity(e);
			}
		}
		
		if (proc.isSetTransitions()) {
			TransitionsDocument.Transitions trs = proc.getTransitions();
			int count = trs.sizeOfTransitionArray();
			for (int i = 0; i < count; i++) {
				TransitionDocument.Transition tr = trs.getTransitionArray(i);
				TransitionDefinition e = new TransitionDefinitionImpl(tr, getActivities());
				addTransition(e);
			}
		}
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * 名称を返す。
	 * 
	 * @return 名称
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * アクティビティ一覧を返す。
	 * 
	 * @return	アクティビティ一覧
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getActivities()
	 */
	public List<ActivityDefinition> getActivities() {
		return activities;
	}

	/**
	 * 期間単位を返す。
	 * 
	 * @return	期間単位
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getDurationUnit()
	 */
	public int getDurationUnit()
	{
		return durationUnit;
	}

	/**
	 * 実行期限を返す。
	 * 
	 * @return	実行期限
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getLimit()
	 */
	public int getLimit()
	{
		return limit;
	}

	/**
	 * 論理座標高さを返す。
	 * 
	 * @return	論理座標高さ
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getLogicalHeight()
	 */
	public int getLogicalHeight()
	{
		return logicalHeight;
	}

	/**
	 * 論理座標幅を返す。
	 * 
	 * @return	論理座標幅
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getLogicalWidth()
	 */
	public int getLogicalWidth()
	{
		return logicalWidth;
	}

	/**
	 * パラメータ一覧を返す。
	 * 
	 * @return	パラメータ一覧
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getParameters()
	 */
	public List<ParameterDefinition> getParameters()
	{
		return parameters;
	}

	/**
	 * 遷移一覧を返す。
	 * 
	 * @return	遷移一覧
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getTransitions()
	 */
	public List<TransitionDefinition> getTransitions()
	{
		return transitions;
	}

	/**
	 * 通知者一覧を返す。
	 * 
	 * @return	通知者一覧
	 * @see jp.co.argo21.nautica.workflow.definition.ProcessDefinition#getResponsibles()
	 */
	public List<String> getResponsibles()
	{
		return responsibles;
	}

	/**
	 * アクセスレベルを返す。
	 * 
	 * @return	アクセスレベル
	 */
	public int getAccessLevel()
	{
		return accessLevel;
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * アクセスレベルを設定する。
	 * 
	 * @param i	アクセスレベル
	 */
	private void setAccessLevel(int i)
	{
		accessLevel = i;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 */
	private void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * 名称を設定する。
	 * 
	 * @param name 名称
	 */
	private void setName(String name)
	{
		this.name = name;
	}

	/**
	 * 期間単位を設定する。
	 * 
	 * @param i	期間単位
	 */
	private void setDurationUnit(int i)
	{
		durationUnit = i;
	}

	/**
	 * 実行期限を設定する。
	 * 
	 * @param i	実行期限
	 */
	private void setLimit(int i)
	{
		limit = i;
	}

	/**
	 * 論理座標高さを設定する。
	 * 
	 * @param i	論理座標高さ
	 */
	private void setLogicalHeight(int i)
	{
		logicalHeight = i;
	}

	/**
	 * 論理座標幅を設定する。
	 * 
	 * @param i	論理座標幅
	 */
	private void setLogicalWidth(int i)
	{
		logicalWidth = i;
	}

	/**
	 * アクティビティを追加する。
	 * 
	 * @param act	アクティビティ
	 */
	private void addActivity(ActivityDefinition act)
	{
		this.activities.add(act);
	}

	/**
	 * パラメータを追加する。
	 * 
	 * @param param	パラメータ
	 */
	private void addParameter(ParameterDefinition param)
	{
		this.parameters.add(param);
	}

	/**
	 * 通知者を追加する。
	 * 
	 * @param r	通知者
	 */
	private void addResponsible(String r)
	{
		this.responsibles.add(r);
	}

	/**
	 * 遷移を追加する。
	 * 
	 * @param tr	遷移
	 */
	private void addTransition(TransitionDefinition tr)
	{
		this.transitions.add(tr);
	}
}
