/* $Id: PackageDefinitionImpl.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.definition.impl;

import java.util.ArrayList;
import java.util.List;

import org.wfmc.x2002.xpdl10.ApplicationDocument;
import org.wfmc.x2002.xpdl10.ApplicationsDocument;
import org.wfmc.x2002.xpdl10.DataFieldDocument;
import org.wfmc.x2002.xpdl10.DataFieldsDocument;
import org.wfmc.x2002.xpdl10.PackageDocument;
import org.wfmc.x2002.xpdl10.PackageHeaderDocument;
import org.wfmc.x2002.xpdl10.ParticipantDocument;
import org.wfmc.x2002.xpdl10.ParticipantsDocument;
import org.wfmc.x2002.xpdl10.RedefinableHeaderDocument;
import org.wfmc.x2002.xpdl10.WorkflowProcessDocument;
import org.wfmc.x2002.xpdl10.WorkflowProcessesDocument;

import jp.co.argo21.nautica.workflow.definition.ApplicationDefinition;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.definition.ParticipantDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.definition.VariableDefinition;
import jp.co.argo21.nautica.workflow.util.IDGenerator;
import jp.co.argo21.nautica.workflow.util.StringUtils;

/**
 * PackageDefinitionインタフェースの実装クラスである。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class PackageDefinitionImpl implements PackageDefinition
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** 未初期化時の仮ID */
	private static final String UNINITIALIZED_ID = "--UNINITIALIZED--"; 

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 作成日 */
	private String createdDate;
	/** バージョン */
	private String version;
	/** 定義者 */
	private String author;
	/** ワークフロー参加者一覧 */
	private List<ParticipantDefinition> participants = new ArrayList<ParticipantDefinition>();
	/** アプリケーション一覧 */
	private List<ApplicationDefinition> applications = new ArrayList<ApplicationDefinition>();
	/** プロセス変数一覧 */
	private List<VariableDefinition> variables = new ArrayList<VariableDefinition>();
	/** プロセス一覧 */
	private List<ProcessDefinition> processes = new ArrayList<ProcessDefinition>();

	/**
	 * パッケージを生成する。
	 * 
	 * @param doc	XmlObject
	 */
	public PackageDefinitionImpl(PackageDocument doc)
	{
		PackageDocument.Package p = doc.getPackage();
		String id = p.getId();
		if (StringUtils.checkNull(id) || id.equals(UNINITIALIZED_ID)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(p.getName());
		
		PackageHeaderDocument.PackageHeader phd = p.getPackageHeader();
		setDescription(phd.getDescription());
		setCreatedDate(phd.getCreated());
		
		if (p.isSetRedefinableHeader()) {
			RedefinableHeaderDocument.RedefinableHeader rhd
				= p.getRedefinableHeader();
			setAuthor(rhd.getAuthor());
			setVersion(rhd.getVersion());
		}
		
		if (p.isSetApplications()) {
			ApplicationsDocument.Applications apps = p.getApplications();
			int count = apps.sizeOfApplicationArray();
			for (int i = 0; i < count; i++) {
				ApplicationDocument.Application app = apps.getApplicationArray(i);
				ApplicationDefinition e = new ApplicationDefinitionImpl(app);
				addApplication(e);
			}
		}
		
		if (p.isSetParticipants()) {
			ParticipantsDocument.Participants pts = p.getParticipants();
			int count = pts.sizeOfParticipantArray();
			for (int i = 0; i < count; i++) {
				ParticipantDocument.Participant pt = pts.getParticipantArray(i);
				ParticipantDefinition e = new ParticipantDefinitionImpl(pt);
				addParticipant(e);
			}
		}
		
		if (p.isSetDataFields()) {
			DataFieldsDocument.DataFields dfs = p.getDataFields();
			int count = dfs.sizeOfDataFieldArray();
			for (int i = 0; i < count; i++) {
				DataFieldDocument.DataField df = dfs.getDataFieldArray(i);
				VariableDefinition e = new VariableDefinitionImpl(df);
				addVariable(e);
			}
		}
		
		if (p.isSetWorkflowProcesses()) {
			WorkflowProcessesDocument.WorkflowProcesses wps = p.getWorkflowProcesses();
			int count = wps.sizeOfWorkflowProcessArray();
			for (int i = 0; i < count; i++) {
				WorkflowProcessDocument.WorkflowProcess proc = wps.getWorkflowProcessArray(i);
				ProcessDefinition e = new ProcessDefinitionImpl(proc);
				addProcess(e);
			}
		}
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * 名称を返す。
	 * 
	 * @return 名称
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * アプリケーション一覧を返す。
	 * 
	 * @return	アプリケーション一覧
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getApplications()
	 */
	public List<ApplicationDefinition> getApplications()
	{
		return applications;
	}

	/**
	 * 作成日を返す。
	 * 
	 * @return	作成日
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getCreatedDate()
	 */
	public String getCreatedDate()
	{
		return createdDate;
	}

	/**
	 * ワークフロー参加者一覧を返す。
	 * 
	 * @return	ワークフロー参加者一覧
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getParticipants()
	 */
	public List<ParticipantDefinition> getParticipants()
	{
		return participants;
	}

	/**
	 * プロセス一覧を返す。
	 * 
	 * @return	プロセス一覧
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getProcesses()
	 */
	public List<ProcessDefinition> getProcesses()
	{
		return processes;
	}

	/**
	 * プロセス変数一覧を返す。
	 * 
	 * @return	プロセス変数一覧
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getVariables()
	 */
	public List<VariableDefinition> getVariables()
	{
		return variables;
	}

	/**
	 * バージョンを返す。
	 * 
	 * @return	バージョン
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getVersion()
	 */
	public String getVersion()
	{
		return version;
	}

	/**
	 * 定義者を返す。
	 * 
	 * @return	定義者
	 * @see jp.co.argo21.nautica.workflow.definition.PackageDefinition#getAuthor()
	 */
	public String getAuthor()
	{
		return author;
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 */
	private void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * 名称を設定する。
	 * 
	 * @param name 名称
	 */
	private void setName(String name)
	{
		this.name = name;
	}

	/**
	 * 作成日を設定する。
	 * 
	 * @param s 作成日
	 */
	private void setCreatedDate(String s)
	{
		createdDate = s;
	}

	/**
	 * バージョンを設定する。
	 * 
	 * @param s バージョン
	 */
	private void setVersion(String s)
	{
		version = s;
	}

	/**
	 * 定義者を設定する。
	 * 
	 * @param s 定義者
	 */
	private void setAuthor(String s)
	{
		author = s;
	}

	/**
	 * アプリケーションを追加する。
	 * 
	 * @param app	アプリケーション
	 */
	private void addApplication(ApplicationDefinition app)
	{
		this.applications.add(app);
	}

	/**
	 * ワークフロー参加者を追加する。
	 * 
	 * @param part	ワークフロー参加者
	 */
	private void addParticipant(ParticipantDefinition part)
	{
		this.participants.add(part);
	}

	/**
	 * プロセスを追加する。
	 * 
	 * @param proc	プロセス
	 */
	private void addProcess(ProcessDefinition proc)
	{
		this.processes.add(proc);
	}

	/**
	 * プロセス変数を追加する。
	 * 
	 * @param var	プロセス変数
	 */
	private void addVariable(VariableDefinition var)
	{
		this.variables.add(var);
	}
}
