/* $Id: MySQLWorkItemDAO.java,v 1.3 2007/12/06 07:47:15 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.WorkItemBean;
import jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidQueryException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.Participant;
import jp.co.argo21.nautica.workflow.wfmc.WorkItem;

/**
 * WorkItemBeanをデータとするMySQL用DAOクラスである。
 * <pre><tt>
 * WORKITEMテーブル
 * +------------------+--------------+------+-----+---------+-------+
 * | Field            | Type         | Null | Key | Default | Extra |
 * +------------------+--------------+------+-----+---------+-------+
 * | WORK_ID          | varchar(64)  |      | PRI |         |       |
 * | PROC_ID          | varchar(64)  |      | MUL |         |       |
 * | ACT_ID           | varchar(64)  |      |     |         |       |
 * | PARTICIPANT_TYPE | int(1)       | YES  |     | NULL    |       |
 * | PARTICIPANT_NAME | varchar(128) | YES  |     | NULL    |       |
 * +------------------+--------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLWorkItemDAO extends AbstractDAO implements WorkItemDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/**　作業項目IDのカラム名 */
	public static final String WORK_ID          = "WORK_ID";
	/**　プロセスIDのカラム名 */
	public static final String PROC_ID          = "PROC_ID";
	/**　アクティビティIDのカラム名 */
	public static final String ACT_ID           = "ACT_ID";
	/**　参加者型のカラム名 */
	public static final String PARTICIPANT_TYPE = "PARTICIPANT_TYPE";
	/**　参加者名のカラム名 */
	public static final String PARTICIPANT_NAME = "PARTICIPANT_NAME";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from WORKITEM where WORK_ID = ? ";

	private static final String SQL_SELECT_BY_PID_AND_AID
	= "select * from WORKITEM where PROC_ID = ? AND ACT_ID = ? ";

	private static final String SQL_SELECT_COUNT_BY_PARTICIPANT
	= "select count(*) from WORKITEM where PARTICIPANT_NAME = ? and PARTICIPANT_TYPE = ? ";

	private static final String SQL_SELECT_BY_PARTICIPANT
	= "select * from WORKITEM where  PARTICIPANT_NAME = ? and PARTICIPANT_TYPE = ? ";

	private static final String SQL_SELECT_WITH_VAR
	= "select distinct w.* from WORKITEM w, VARIABLE v ";

	private static final String SQL_SELECT
	= "select distinct w.* from WORKITEM w ";

	private static final String SQL_INSERT
	= "insert into WORKITEM ("
		+ WORK_ID + ", "
		+ PROC_ID + ", "
		+ ACT_ID + ", "
		+ PARTICIPANT_TYPE + ", "
		+ PARTICIPANT_NAME + ") values (?,?,?,?,?)";

	private static final String SQL_UPDATE_PARTICIPANT
	= "update WORKITEM set PARTICIPANT_TYPE = ?, PARTICIPANT_NAME = ? where WORK_ID = ? and PARTICIPANT_TYPE = ? and PARTICIPANT_NAME = ? ";

	private static final String SQL_DELETE
	= "delete from WORKITEM where WORK_ID = ? ";

	private static final String SQL_DELETE_BY_PID_AND_AID
	= "delete from WORKITEM where PROC_ID = ? AND ACT_ID = ? ";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLWorkItemDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * 作業項目IDで作業項目を取得する。
	 *
	 * @param wid 作業項目ID
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#findByPrimaryKey(java.lang.String)
	 */
	public WorkItemBean findByPrimaryKey(String wid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, wid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			WorkItemBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * プロセスIDとアクティビティIDで作業項目を取得する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#findByProcessIDANDActivityID(java.lang.String, java.lang.String)
	 */
	public WorkItemBean findByProcessIDANDActivityID(String pid, String aid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PID_AND_AID);
		try {
			st.setString(1, pid);
			st.setString(2, aid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			WorkItemBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * ワークフロー参加者名と型で作業項目の個数を取得する。
	 *
	 * @param name ワークフロー参加者名
	 * @param type ワークフロー参加者型
	 * @return 作業項目の個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#findCountByParticipant(java.lang.String, int)
	 */
	public int findCountByParticipant(String name, int type)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_COUNT_BY_PARTICIPANT);
		try {
			st.setString(1, name);
			st.setInt(2, type);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			int count = 0;
			if (rs.next()) {
				count = rs.getInt(1);
			}
			return count;
		} finally {
			st.close();
		}
	}
	
	/**
	 * ワークフロー参加者名と型で作業項目を取得する。
	 *
	 * @param name ワークフロー参加者名
	 * @param type ワークフロー参加者型
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#findByParticipant(java.lang.String, int)
	 */
	public WorkItemBean[] findByParticipant(String name, int type)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PARTICIPANT);
		try {
			st.setString(1, name);
			st.setInt(2, type);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			List<WorkItemBean> beans = new ArrayList<WorkItemBean>();
			while (rs.next()) {
				WorkItemBean bean = createBean(rs, "w.");
				beans.add(bean);
			}
			return (WorkItemBean[])beans.toArray(new WorkItemBean[beans.size()]);
		} finally {
			st.close();
		}
	}
	
	/**
	 * SQL部分文字列にしたがって、作業項目を取得する。
	 *
	 * @param s SQL部分文字列
	 * @param hasAttrbuteFilter 属性フィルターを含む場合true
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#findBySQLPart(java.lang.String, boolean)
	 */
	public WorkItemBean[] findBySQLPart(String s, boolean hasAttrbuteFilter)
	throws SQLException, DAOException
	{
		PreparedStatement st = null;
		if (hasAttrbuteFilter) {
			st = getConnection().prepareStatement(SQL_SELECT_WITH_VAR + s);
		} else {
			st = getConnection().prepareStatement(SQL_SELECT + s);
		}
		try {

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/

			ResultSet rs = st.executeQuery();
			
			List<WorkItemBean> beans = new ArrayList<WorkItemBean>();
			while (rs.next()) {
				WorkItemBean bean = createBean(rs, "w.");
				beans.add(bean);
			}
			return (WorkItemBean[])beans.toArray(new WorkItemBean[beans.size()]);
		} finally {
			st.close();
		}
	}

	/**
	 * 作業項目一覧の取得を行う。
	 *
	 * @param filter 検索用フィルター
	 * @param userResource ユーザ情報
	 * @param orgResources 組織情報
	 * @param roleResources ロール情報
	 * @return 作業項目一覧
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws InvalidQueryException SQL部分文字列の生成に失敗した場合
	 */
	public WorkItem[] findByFilter(Filter filter, Participant userResource, List<Participant> orgResources, List<Participant> roleResources)
	throws SQLException, DAOException, InvalidFilterException, InvalidQueryException
	{
		WorkItemFilterConverter conv = new WorkItemFilterConverter();
		String filterPart = conv.convertToSQL(filter);
		String participantPart = getParticipantSQLPart(userResource, orgResources, roleResources);
		StringBuffer sqlPart = new StringBuffer();
		boolean filterExist = (! filterPart.equals(""));
		boolean hasAttrFilter = conv.hasAttributeFilter();

		sqlPart.append(" where ").append(participantPart);
		if (filterExist) {
			if (hasAttrFilter) {
				sqlPart.append(" and ").append(filterPart)
				.append(" and (w.PROC_ID = v.PROC_ID)");
			} else {
				sqlPart.append(" and ").append(filterPart);
			}
		}

		WorkItem[] items = findBySQLPart(sqlPart.toString(), hasAttrFilter);
		
		return items;
	}

	/**
	 * 対応する作業項目のチェックを行い、それを返す。
	 *
	 * @param wid 作業項目ID
	 * @param userResource ユーザ情報
	 * @param orgResources 組織情報
	 * @param roleResources ロール情報
	 * @return　作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidWorkItemException 作業項目が特定できない場合
	 * @throws InvalidQueryException SQL部分文字列の生成に失敗した場合
	 */
	public WorkItem findByWorkItemID(String wid, Participant userResource, List<Participant> orgResources, List<Participant> roleResources)
	throws DAOException, SQLException, InvalidWorkItemException, InvalidQueryException 
	{
		String participantPart = getParticipantSQLPart(userResource, orgResources, roleResources);

		StringBuffer sqlPart
			= new StringBuffer().append(" where w.WORK_ID = '")
				.append(wid).append("' and ").append(participantPart);

		WorkItem[] items = findBySQLPart(sqlPart.toString(), false);
		
		if (items.length != 1) {
			// 作業項目が特定できません。
			String E0181 = StringManager.get("E0181");
			throw new InvalidWorkItemException(E0181);
		}
		return items[0];
	}
	
	/**
	 * 作業項目を挿入する。
	 *
	 * @param bean 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.WorkItemBean)
	 */
	public void insert(WorkItemBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getID());
			st.setString(2, bean.getProcessID());
			st.setString(3, bean.getActivityID());
			st.setInt(4, bean.getParticipantType());
			st.setString(5, bean.getParticipantName());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 作業項目の挿入に失敗しました。
				String errMsg = StringManager.get("E0035");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 作業項目のワークフロー参加者を更新する。
	 *
	 * @param wid 作業項目ID
	 * @param src 更新前のワークフロー参加者
	 * @param target 更新後のワークフロー参加者
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#updateParticipant(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Participant, jp.co.argo21.nautica.workflow.wfmc.Participant)
	 */
	public void updateParticipant(String wid, Participant src, Participant target)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_PARTICIPANT);
		try {
			if (target == null) {
				st.setNull(1, Types.INTEGER);
				st.setNull(2, Types.VARCHAR);
				st.setString(3, wid);
				st.setInt(4, src.getType());
				st.setString(5, src.getID());
			} else {
				st.setInt(1, target.getType());
				st.setString(2, target.getID());
				st.setString(3, wid);
				st.setInt(4, src.getType());
				st.setString(5, src.getID());
			}

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 作業項目の担当者の変更に失敗しました。
				String errMsg = StringManager.get("E0036");
				throw new DAOException(errMsg);
			}
		} catch (WorkflowException ex) {
			// 作業項目の担当者の変更に失敗しました。
			String errMsg = StringManager.get("E0036");
			throw new DAOException(errMsg, ex);
		} finally {
			st.close();
		}
	}
	
	/**
	 * 作業項目を削除する。
	 *
	 * @param wid 作業項目ID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#delete(java.lang.String)
	 */
	public void delete(String wid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_DELETE);
		try {
			st.setString(1, wid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 作業項目の削除に失敗しました。
				String errMsg = StringManager.get("E0034");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}

	/**
	 * プロセスIDとアクティビティIDで作業項目を取得する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.WorkItemDAO#deleteByProcessIDANDActivityID(java.lang.String, java.lang.String)
	 */
	public void deleteByProcessIDANDActivityID(String pid, String aid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_DELETE_BY_PID_AND_AID);
		try {
			st.setString(1, pid);
			st.setString(2, aid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 作業項目の削除に失敗しました。
				String errMsg = StringManager.get("E0034");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットから作業項目を作成する。
	 *
	 * @param rs 結果セット
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 */
	private WorkItemBean createBean(ResultSet rs)
	throws SQLException
	{
		WorkItemBean bean = new WorkItemBean();

		bean.setID(rs.getString(WORK_ID));
		bean.setProcessID(rs.getString(PROC_ID));
		bean.setActivityID(rs.getString(ACT_ID));
		bean.setParticipantType(rs.getInt(PARTICIPANT_TYPE));
		bean.setParticipantName(rs.getString(PARTICIPANT_NAME));
		
		return bean;
	}
	
	/**
	 * 結果セットから作業項目を作成する。
	 *
	 * @param rs 結果セット
	 * @param prefix 前置文字列
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 */
	private WorkItemBean createBean(ResultSet rs, String prefix)
	throws SQLException
	{
		WorkItemBean bean = new WorkItemBean();

		bean.setID(rs.getString(prefix + WORK_ID));
		bean.setProcessID(rs.getString(prefix + PROC_ID));
		bean.setActivityID(rs.getString(prefix + ACT_ID));
		bean.setParticipantType(rs.getInt(prefix + PARTICIPANT_TYPE));
		bean.setParticipantName(rs.getString(prefix + PARTICIPANT_NAME));
		
		return bean;
	}
	
	/**
	 * ワークフロー参加者に対応したSQL部分文字列を生成する。
	 *
	 * @param userResource ユーザ情報
	 * @param orgResources 組織情報
	 * @param roleResources ロール情報
	 * @return SQL部分文字列
	 * @throws InvalidQueryException SQL部分文字列の生成に失敗した場合
	 */
	private String getParticipantSQLPart(Participant userResource, List<Participant> orgResources, List<Participant> roleResources)
	throws InvalidQueryException
	{
		StringBuffer sb = new StringBuffer();
		try {
			sb.append("((w.PARTICIPANT_TYPE = ").append(userResource.getType())
				.append(" and w.PARTICIPANT_NAME = '").append(userResource.getID()).append("')");

			if (orgResources != null && orgResources.size() > 0) {
				sb.append("or (w.PARTICIPANT_TYPE = ").append(Participant.ORGANIZATIONAL_UNIT);
				Iterator<Participant> it = orgResources.iterator();
				Participant p = (Participant)it.next();
				sb.append(" and (w.PARTICIPANT_NAME = '").append(p.getID()).append("' ");
				while (it.hasNext()) {
					p = (Participant)it.next();
					sb.append("or w.PARTICIPANT_NAME = '").append(p.getID()).append("' ");
				}
				sb.append("))");
			}

			if (roleResources != null && roleResources.size() > 0) {
				sb.append("or (w.PARTICIPANT_TYPE = ").append(Participant.ROLE);
				Iterator<Participant> it = roleResources.iterator();
				Participant p = (Participant)it.next();
				sb.append(" and (w.PARTICIPANT_NAME = '").append(p.getID()).append("' ");
				while (it.hasNext()) {
					p = (Participant)it.next();
					sb.append("or w.PARTICIPANT_NAME = '").append(p.getID()).append("' ");
				}
				sb.append("))");
			}
			sb.append(")");
			
			return sb.toString();
		} catch (WorkflowException ex) {
			// 作業項目が特定できません。
			String E0181 = StringManager.get("E0181");
			throw new InvalidQueryException(E0181, ex);
		}
	}
}
