/* $Id: MySQLTargetProcessDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.TargetProcessBean;
import jp.co.argo21.nautica.workflow.dataaccess.TargetProcessDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * TargetProcessBeanをデータとするMySQL用DAOクラスである。
 * <pre><tt>
 * ETAPROCESSテーブル
 * +--------------+-------------+------+-----+---------+-------+
 * | Field        | Type        | Null | Key | Default | Extra |
 * +--------------+-------------+------+-----+---------+-------+
 * | ENGINE_ID    | varchar(64) |      | PRI |         |       |
 * | PROCESS_ID   | varchar(64) |      | PRI |         |       |
 * | ROOT_PROC_ID | varchar(64) |      |     |         |       |
 * | ROOT_ACT_ID  | varchar(64) |      |     |         |       |
 * +--------------+-------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLTargetProcessDAO extends AbstractDAO implements TargetProcessDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** エンジンIDのカラム名 */
	public static final String ENGINE_ID    = "ENGINE_ID";
	/** プロセスIDのカラム名 */
	public static final String PROCESS_ID   = "PROCESS_ID";
	/** ルートプロセスIDのカラム名 */
	public static final String ROOT_PROC_ID = "ROOT_PROC_ID";
	/** ルートアクティビティIDのカラム名 */
	public static final String ROOT_ACT_ID  = "ROOT_ACT_ID";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from ETAPROCESS where ENGINE_ID = ? AND PROCESS_ID = ? ";

	private static final String SQL_SELECT_BY_PROC_ID
	= "select * from ETAPROCESS where PROCESS_ID = ? ";

	private static final String SQL_SELECT_BY_ROOT_ID
	= "select * from ETAPROCESS where ROOT_PROC_ID = ? AND ROOT_ACT_ID = ? ";

	private static final String SQL_INSERT
	= "insert into ETAPROCESS ("
		+ ENGINE_ID + ", "
		+ PROCESS_ID + ", "
		+ ROOT_PROC_ID + ", "
		+ ROOT_ACT_ID + ") values (?,?,?,?)";

	private static final String SQL_DELETE
	= "delete from ETAPROCESS where ENGINE_ID = ? AND PROCESS_ID = ? ";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLTargetProcessDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * エンジンIDと起動先プロセスIDで起動先プロセス情報を取得する。
	 *
	 * @param eid エンジンID
	 * @param pid プロセスID
	 * @return 起動先プロセス情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.TargetProcessDAO#findByPrimaryKey(java.lang.String, java.lang.String)
	 */
	public TargetProcessBean findByPrimaryKey(String eid, String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, eid);
			st.setString(2, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			TargetProcessBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * 起動先プロセスIDで起動先プロセス情報を取得する。
	 *
	 * @param pid プロセスID
	 * @return 起動先プロセス情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.TargetProcessDAO#findByProcessID(java.lang.String)
	 */
	public TargetProcessBean findByProcessID(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PROC_ID);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			TargetProcessBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * 起動元プロセスIDと起動元アクティビティIDで起動先プロセス情報を取得する。
	 *
	 * @param rootpid 起動元プロセスID
	 * @param rootaid 起動元アクティビティID
	 * @return 起動先プロセス情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.TargetProcessDAO#findByRootID(java.lang.String, java.lang.String)
	 */
	public TargetProcessBean findByRootID(String rootpid, String rootaid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_ROOT_ID);
		try {
			st.setString(1, rootpid);
			st.setString(2, rootaid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			TargetProcessBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * 起動先プロセス情報を挿入する。
	 *
	 * @param bean 起動先プロセス情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.TargetProcessDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.TargetProcessBean)
	 */
	public void insert(TargetProcessBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getEngineID());
			st.setString(2, bean.getProcessID());
			st.setString(3, bean.getRootProcID());
			st.setString(4, bean.getRootActID());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 外部プロセス起動先情報の挿入に失敗しました。
				String errMsg = StringManager.get("E0028");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}

	/**
	 * 起動先プロセス情報を削除する。
	 *
	 * @param eid エンジンID
	 * @param pid プロセスID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.TargetProcessDAO#delete(java.lang.String, java.lang.String)
	 */
	public void delete(String eid, String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_DELETE);
		try {
			st.setString(1, eid);
			st.setString(2, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 外部プロセス起動先情報の削除に失敗しました。
				String errMsg = StringManager.get("E0027");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットから起動先プロセス情報を作成する。
	 *
	 * @param rs 結果セット
	 * @return 起動先プロセス情報
	 * @throws SQLException 任意のSQL例外
	 */
	private TargetProcessBean createBean(ResultSet rs)
	throws SQLException
	{
		TargetProcessBean bean = new TargetProcessBean();

		bean.setEngineID(rs.getString(ENGINE_ID));
		bean.setProcessID(rs.getString(PROCESS_ID));
		bean.setRootProcID(rs.getString(ROOT_PROC_ID));
		bean.setRootActID(rs.getString(ROOT_ACT_ID));
		
		return bean;
	}
}
