/* $Id: MySQLSequencerDAO.java,v 1.2 2007/11/05 02:43:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.SequencerDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;

/**
 * ユニークIDを生成するためのシーケンサを提供するクラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLSequencerDAO extends AbstractDAO implements SequencerDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** シーケンス番号のカラム名 */
	public static final String SEQ  = "SEQ";
	/** シーケンス名のカラム名 */
	public static final String NAME = "NAME";

	/** 新しいシーケンサを追加するためのSQL文 */
	private static final String SQL_INSERT_SEQUENCER
		= "INSERT INTO KEYSEQ (NAME, SEQ) VALUES (?, 1) ON DUPLICATE KEY UPDATE SEQ = SEQ + 1 ";
	/** シーケンサの値を取得するためのSQL文 */
	private static final String SQL_GET_SEQUENCER
		= "SELECT SEQ FROM KEYSEQ WHERE NAME = ? ";

	/**
	 * シーケンス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLSequencerDAO(Connection conn)
	{
		super(conn);
	}

	/**
	 * 次のシーケンス値をintの範囲内で取得する。
	 *
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	public int nextForInt(String seqName) throws SQLException
	{
		long seq = next(seqName);
		if (seq > Integer.MAX_VALUE) {
			throw new ArithmeticException("Value is out of INTEGER range.");
		}
		return (int)seq;
	}

	/**
	 * 次のシーケンス値を取得する。
	 *
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	public synchronized long next(String seqName) throws SQLException
	{
		Connection conn = getConnection();
		updateValue(conn, seqName);
		long seq = currentValue(conn, seqName);
		return seq;
	}

	/**
	 * 次のシーケンス値を取得する。
	 *
	 * @param conn    JDBC接続
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	public void updateValue(Connection conn, String seqName) throws SQLException
	{
		PreparedStatement statement = conn.prepareStatement(SQL_INSERT_SEQUENCER);
		try {
			statement.setString(1, seqName);

			/************** DEBUG **************/
			eLog.debug(statement.toString());
			/***********************************/
			statement.executeUpdate();
		} finally {
			statement.close();
		}
	}

	/**
	 * 現在のシーケンス値を取得する。
	 *
	 * @param conn    JDBC接続
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	private long currentValue(Connection conn, String seqName)
	throws SQLException
	{
		PreparedStatement statement = conn.prepareStatement(SQL_GET_SEQUENCER);
		try {
			statement.setString(1, seqName);

			/************** DEBUG **************/
			eLog.debug(statement.toString());
			/***********************************/

			long seq = -1L;
			ResultSet rs = statement.executeQuery();
			if (rs.next()) {
				seq = rs.getLong(SEQ);
			}
			rs.close();

			return seq;
		} finally {
			statement.close();
		}
	}
}
