/* $Id: ActivityFilterConverter.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import jp.co.argo21.nautica.workflow.filter.AttributeFilter;
import jp.co.argo21.nautica.workflow.filter.FilterConverter;
import jp.co.argo21.nautica.workflow.filter.GroupFilter;
import jp.co.argo21.nautica.workflow.filter.NameFilter;
import jp.co.argo21.nautica.workflow.filter.StateFilter;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * アクティビティ検索用のフィルターからSQLのwhere句を生成するコンバータ。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ActivityFilterConverter implements FilterConverter
{
	private boolean hasAttributeFilter;

	/**
	 * コンバータを生成する。
	 *
	 */
	public ActivityFilterConverter() {
	    // 処理なし。
	}
	
	/**
	 * アクティビティ検索用のフィルターからSQLのwhere句を生成する
	 *
	 * @param filter フィルター
	 * @return SQLのwhere句
	 * @see jp.co.argo21.nautica.workflow.filter.FilterConverter#convertToSQL(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public String convertToSQL(Filter filter) throws InvalidFilterException
	{
		String sql = "";
		
		if (filter == null) {
			//NOP
		} else if (filter instanceof AttributeFilter) {
			sql = convertToSQL((AttributeFilter)filter);
		} else if (filter instanceof StateFilter) {
			sql = convertToSQL((StateFilter)filter);
		} else if (filter instanceof GroupFilter) {
			sql = convertToSQL((GroupFilter)filter);
		} else {
			// 無効なフィルターが指定されました。
			String errMsg = StringManager.get("E0004");
			throw new InvalidFilterException(errMsg);
		}

		if (hasAttributeFilter) {
			sql += " AND a.PROC_ID = v.PROC_ID ";
		}

		return sql;
	}
	
	/**
	 * AttributeFilterをSQL部分文字列に変換する。
	 *
	 * @param filter AttributeFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(AttributeFilter filter)
	throws InvalidFilterException
	{
		String name = filter.getFilterName();
		if (name == null) {
			// 無効なフィルターが指定されました。
			String errMsg = StringManager.get("E0004");
			throw new InvalidFilterException(errMsg);
		} else {
			name = "'" + name + "'";
		}
		
		hasAttributeFilter = true;
		
		String fs = filter.getFilterString();
		if (fs == null) {
			fs = "NULL";
		} else {
			fs = "'" + fs + "'";
		}

		String sql =  "( v." + MySQLVariableDAO.NAME
		+ " = " + name
		+ " AND v." + MySQLVariableDAO.VALUE
		+ getComparisonString(filter.getComparison())
		+ fs + ")";
		return sql;
	}
	
	/**
	 * StateFilterをSQL部分文字列に変換する。
	 *
	 * @param filter StateFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(StateFilter filter)
	throws InvalidFilterException
	{
		String fs = filter.getFilterString();
		if (fs == null) {
			// 無効なフィルターが指定されました。
			String errMsg = StringManager.get("E0004");
			throw new InvalidFilterException(errMsg);
		} else {
			fs = "'a." + fs + "'";
		}

		return "a." + MySQLActivityDAO.ACT_STATE
			+ " = " +  filter.getFilterString();
	}
	
	/**
	 * GroupFilterをSQL部分文字列に変換する。
	 *
	 * @param filter GroupFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(GroupFilter filter)
	throws InvalidFilterException
	{
		Filter[] children = filter.getChild();
		
		int count = children.length;
		if (count == 0) return "";
		
		String s = convertToSQL(children[0]);
		
		for (int i = 1; i < count; i++) {
			s += " AND " + convertToSQL(children[0]);
		}
		
		return s;
	}
	
	/**
	 * 演算子を文字列に変換する。
	 *
	 * @param comparison 演算子
	 * @return 部分文字列
	 */
	private String getComparisonString(int comparison)
	{
		switch (comparison) {
		case NameFilter.EQUAL:
			return " = ";
		case NameFilter.GREATER_EQUAL:
			return " >= ";
		case NameFilter.GREATER_THAN:
			return " > ";
		case NameFilter.LESS_EQUAL:
			return " <= ";
		case NameFilter.LESS_THAN:
			return " < ";
		case NameFilter.NOT_EQUAL:
			return " <> ";
		case NameFilter.LIKE:
			return " LIKE ";
		default:
			return " <> ";
		}
	}
	
	/**
	 * 属性フィルターが含まれているかどうかを返す。
	 *
	 * @return 含まれている場合は、true
	 */
	public boolean hasAttributeFilter()
	{
		return hasAttributeFilter;
	}
}
