/* $Id: WorkItemFilterConverter.java,v 1.1 2007/11/13 08:51:28 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import jp.co.argo21.nautica.workflow.filter.AttributeFilter;
import jp.co.argo21.nautica.workflow.filter.FilterConverter;
import jp.co.argo21.nautica.workflow.filter.GroupFilter;
import jp.co.argo21.nautica.workflow.filter.NameFilter;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * 作業項目検索用のフィルターからSQLのwhere句を生成するコンバータ。
 * 作業項目検索に使用できるものは以下のフィルターである。
 * <ul>
 * <li>AttributeFilter</li>
 * <li>GroupFilter</li>
 * </ul>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class WorkItemFilterConverter implements FilterConverter
{
	private boolean hasAttributeFilter;

	/**
	 * コンバータを生成する。
	 *
	 */
	public WorkItemFilterConverter() {
	    // 処理なし。
	}
	
	/**
	 * 作業項目検索用のフィルターからSQLのwhere句を生成する
	 *
	 * @param filter フィルター
	 * @return SQLのwhere句
	 * @see jp.co.argo21.nautica.workflow.filter.FilterConverter#convertToSQL(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public String convertToSQL(Filter filter) throws InvalidFilterException
	{
		if (filter == null) {
			return "";
		} else if (filter instanceof AttributeFilter) {
			return convertToSQL((AttributeFilter)filter);
		} else if (filter instanceof GroupFilter) {
			return convertToSQL((GroupFilter)filter);
		} else {
			// 無効なフィルターが指定されました。
			String errMsg = StringManager.get("E0004");
			throw new InvalidFilterException(errMsg);
		}
	}
	
	/**
	 * AttributeFilterをSQL部分文字列に変換する。
	 *
	 * @param filter AttributeFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(AttributeFilter filter)
	throws InvalidFilterException
	{
		String name = filter.getFilterName();
		if (name == null) {
			// 無効なフィルターが指定されました。
			String errMsg = StringManager.get("E0004");
			throw new InvalidFilterException(errMsg);
		} else {
			name = "'" + name + "'";
		}
		
		hasAttributeFilter = true;
		
		String fs = filter.getFilterString();
		if (fs == null) {
			fs = "NULL";
		} else {
			fs = "'" + fs + "'";
		}

		String sql =  "( v." + DerbyVariableDAO.NAME
		+ " = " + name
		+ " AND v." + DerbyVariableDAO.VALUE
		+ getComparisonString(filter.getComparison())
		+ fs + ")";
		return sql;
	}
	
	/**
	 * GroupFilterをSQL部分文字列に変換する。
	 *
	 * @param filter GroupFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(GroupFilter filter)
	throws InvalidFilterException
	{
		Filter[] children = filter.getChild();
		
		int count = children.length;
		if (count == 0) return "";
		
		String s = convertToSQL(children[0]);
		
		for (int i = 1; i < count; i++) {
			s += " AND " + convertToSQL(children[i]);
		}
		
		return s;
	}
	
	/**
	 * 演算子を文字列に変換する。
	 *
	 * @param comparison 演算子
	 * @return 部分文字列
	 */
	private String getComparisonString(int comparison)
	{
		switch (comparison) {
		case NameFilter.EQUAL:
			return " = ";
		case NameFilter.GREATER_EQUAL:
			return " >= ";
		case NameFilter.GREATER_THAN:
			return " > ";
		case NameFilter.LESS_EQUAL:
			return " <= ";
		case NameFilter.LESS_THAN:
			return " < ";
		case NameFilter.NOT_EQUAL:
			return " <> ";
		case NameFilter.LIKE:
			return " LIKE ";
		default:
			return " <> ";
		}
	}
	
	/**
	 * 属性フィルターが含まれているかどうかを返す。
	 *
	 * @return 含まれている場合は、true
	 */
	public boolean hasAttributeFilter()
	{
		return hasAttributeFilter;
	}
}
