/* $Id: DerbyVariableDAO.java,v 1.5 2007/12/05 06:19:57 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.VariableBean;
import jp.co.argo21.nautica.workflow.dataaccess.VariableDAO;
import jp.co.argo21.nautica.workflow.dataaccess.VariableHistoryBean;
import jp.co.argo21.nautica.workflow.dataaccess.VariableHistoryDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * VariableBeanをデータとするDerby用DAOクラスである。
 * <pre><tt>
 * WF_VARIABLEテーブル
 * +-------------+--------------+------+-----+---------+-------+
 * | Field       | Type         | Null | Key | Default | Extra |
 * +-------------+--------------+------+-----+---------+-------+
 * | VAR_ID      | varchar(64)  |      | PRI |         |       |
 * | NAME        | varchar(128) |      |     |         |       |
 * | PROC_ID     | varchar(64)  |      | MUL |         |       |
 * | VAR_DEF_ID  | varchar(64)  |      |     |         |       |
 * | TYPE        | int(1)       |      |     | 0       |       |
 * | VALUE       | text         |      |     |         |       |
 * | UPDATE_DATE | datetime     | YES  |     | NULL    |       |
 * +-------------+--------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  knakata(Argo 21, Corp.)
 * @version $Revision: 1.5 $
 * @since   Nautica Workflow 1.0
 */
public class DerbyVariableDAO extends AbstractDAO implements VariableDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** プロセス変数IDのカラム名 */
	public static final String VAR_ID      = "VAR_ID";
	/** プロセス変数名のカラム名 */
	public static final String NAME        = "NAME";
	/** プロセスIDのカラム名 */
	public static final String PROC_ID     = "PROC_ID";
	/** プロセス変数定義IDのカラム名 */
	public static final String VAR_DEF_ID  = "VAR_DEF_ID";
	/** プロセス変数タイプのカラム名 */
	public static final String TYPE        = "TYPE";
	/** プロセス変数値のカラム名 */
	public static final String VALUE       = "VALUE";
	/** 更新日時のカラム名 */
	public static final String UPDATE_DATE = "UPDATE_DATE";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from WF_VARIABLE where PROC_ID = ? AND VAR_ID = ? ";

	private static final String SQL_SELECT_BY_DEFINITION_ID
	= "select * from WF_VARIABLE where PROC_ID = ? AND VAR_DEF_ID = ? ";

	private static final String SQL_SELECT_BY_PROCEESS_ID
	= "select * from WF_VARIABLE where PROC_ID = ? ";

	private static final String SQL_SELECT_BY_PROCEESS_ID_AND_NAME
	= "select * from WF_VARIABLE where PROC_ID = ? AND NAME = ? ";

	private static final String SQL_INSERT
	= "insert into WF_VARIABLE ("
		+ VAR_ID + ", "
		+ NAME + ", "
		+ PROC_ID + ", "
		+ VAR_DEF_ID + ", "
		+ TYPE + ", "
		+ VALUE + ", "
		+ UPDATE_DATE + ") values (?,?,?,?,?,?,?)";

	private static final String SQL_UPDATE
	= "update WF_VARIABLE set "
		+ VALUE + " = ?, "
		+ UPDATE_DATE + " = ? where PROC_ID = ? AND NAME = ? AND TYPE = ? ";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	DerbyVariableDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * プロセスIDとプロセス変数IDでプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param vid プロセス変数ID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#findByPrimaryKey(java.lang.String, java.lang.String)
	 */
	public VariableBean findByPrimaryKey(String pid, String vid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, pid);
			st.setString(2, vid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			VariableBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}


	/**
	 * プロセスIDとプロセス変数定義IDでプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param vdid プロセス変数定義ID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#findByDefinitionID(java.lang.String, java.lang.String)
	 */
	public VariableBean findByDefinitionID(String pid, String vdid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_DEFINITION_ID);
		try {
			st.setString(1, pid);
			st.setString(2, vdid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			VariableBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * プロセスIDとプロセス変数名でプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param name プロセス変数名
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#findByName(java.lang.String, java.lang.String)
	 */
	public VariableBean findByName(String pid, String name)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PROCEESS_ID_AND_NAME);
		try {
			st.setString(1, pid);
			st.setString(2, name);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			VariableBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * プロセスIDとプロセス変数IDでプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#findByProcessID(java.lang.String)
	 */
	public VariableBean[] findByProcessID(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PROCEESS_ID);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();

			List<VariableBean> beans = new ArrayList<VariableBean>();
			while (rs.next()) {
				VariableBean bean = createBean(rs);
				beans.add(bean);
			}
			return (VariableBean[])beans.toArray(new VariableBean[beans.size()]);
		} finally {
			st.close();
		}
	}
	
	/**
	 * リストとしてプロセスIDを収録する。
	 *
	 * @param pid プロセスID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#findByProcessID(java.lang.String)
	 */
	public List<VariableBean> findByProcessIDAsList(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PROCEESS_ID);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();

			List<VariableBean> beans = new ArrayList<VariableBean>();
			while (rs.next()) {
				VariableBean bean = createBean(rs);
				beans.add(bean);
			}
			return beans;
		} finally {
			st.close();
		}
	}
	
	/**
	 * プロセスIDとSQL部分文字列でプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param sqlPart SQL部分文字列
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#findBySQLPart(java.lang.String, java.lang.String)
	 */
	public VariableBean[] findBySQLPart(String pid, String sqlPart)
	throws SQLException, DAOException
	{
		String sql = SQL_SELECT_BY_PROCEESS_ID;
		if (sqlPart != null && sqlPart.trim().equals("") == false) {
			sql += " AND " + sqlPart;
		}
		PreparedStatement st = getConnection().prepareStatement(sql);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();

			List<VariableBean> beans = new ArrayList<VariableBean>();
			while (rs.next()) {
				VariableBean bean = createBean(rs);
				beans.add(bean);
			}
			return (VariableBean[])beans.toArray(new VariableBean[beans.size()]);
		} finally {
			st.close();
		}
	}

	/**
	 * フィルターで特定されたプロセス変数を返す。
	 *
	 * @param filter フィルター
	 * @param pid プロセスID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public VariableBean[] findByFilter(Filter filter, String pid)
	throws DAOException, SQLException, InvalidFilterException
	{
		String sqlPart = new VariableFilterConverter().convertToSQL(filter);
		VariableBean[] beans = findBySQLPart(pid, sqlPart);
		
		return beans;
	}
	
	/**
	 * プロセス変数を挿入する。
	 *
	 * @param bean プロセス変数
	 * @param actid アクティビティID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.VariableBean, java.lang.String)
	 */
	public void insert(VariableBean bean, String actid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getID());
			st.setString(2, bean.getName());
			st.setString(3, bean.getProcessID());
			st.setString(4, bean.getVariableDefinitionID());
			st.setInt(5, bean.getType());
			st.setString(6, bean.getValue());
			st.setTimestamp(7, bean.getUpdateDate());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// プロセス変数の挿入に失敗しました。
				String errMsg = StringManager.get("E0029");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}

		try {
			WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
			VariableHistoryDAO ahDAO = factory.getVariableHistoryDAO();
			VariableHistoryBean history = new VariableHistoryBean();
			history.setVarID(bean.getID());
			history.setProcID(bean.getProcessID());
			history.setVarDefID(bean.getVariableDefinitionID());
			history.setType(bean.getType());
			history.setPrevValue(null);
			history.setNewValue(bean.getValue());
			history.setUpdateDate(bean.getUpdateDate());
			history.setReason("NEW VAR CREATION");
			history.setActivityID(actid);
			ahDAO.insert(history);
		} catch (SQLException ex) {
			throw ex;
		} catch (DAOException ex) {
			throw ex;
		} catch (Exception ex) {
			// プロセス変数変更履歴の変更に失敗しました。
			String errMsg = StringManager.get("E0030");
			throw new DAOException(errMsg, ex);
		}
	}
	
	/**
	 * プロセス変数を更新する。
	 *
	 * @param bean プロセス変数
	 * @param value 値の文字列表現
	 * @param actid アクティビティID
	 * @param ts タイムスタンプ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableDAO#update(jp.co.argo21.nautica.workflow.dataaccess.VariableBean, java.lang.String, java.lang.String, java.sql.Timestamp)
	 */
	public void update(VariableBean bean, String value, String actid, Timestamp ts)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE);
		try {
			st.setString(1, value);
			st.setTimestamp(2, ts);
			st.setString(3, bean.getProcessID());
			st.setString(4, bean.getName());
			st.setInt(5, bean.getType());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// プロセス変数の更新に失敗しました。
				String errMsg = StringManager.get("E0031");
				throw new DAOException(errMsg);
			}
			bean.setValue(value);
			bean.setUpdateDate(ts);
		} finally {
			st.close();
		}

		try {
			WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
			VariableHistoryDAO ahDAO = factory.getVariableHistoryDAO();
			VariableHistoryBean history = new VariableHistoryBean();
			history.setVarID(bean.getID());
			history.setProcID(bean.getProcessID());
			history.setVarDefID(bean.getVariableDefinitionID());
			history.setType(bean.getType());
			history.setPrevValue(null);
			history.setNewValue(bean.getValue());
			history.setUpdateDate(bean.getUpdateDate());
			history.setActivityID(actid);
			String msg = "VARIABLE CHANGED";
			history.setReason(msg);
			ahDAO.insert(history);
		} catch (SQLException ex) {
			throw ex;
		} catch (DAOException ex) {
			throw ex;
		} catch (Exception ex) {
			// プロセス変数変更履歴の変更に失敗しました。
			String errMsg = StringManager.get("E0030");
			throw new DAOException(errMsg, ex);
		}
	}
	
	/**
	 * 結果セットからプロセス変数を作成する。
	 *
	 * @param rs 結果セット
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 */
	private VariableBean createBean(ResultSet rs)
	throws SQLException
	{
		VariableBean bean = new VariableBean();

		bean.setID(rs.getString(VAR_ID));
		bean.setName(rs.getString(NAME));
		bean.setProcessID(rs.getString(PROC_ID));
		bean.setVariableDefinitionID(rs.getString(VAR_DEF_ID));
		bean.setType(rs.getInt(TYPE));
		bean.setValue(rs.getString(VALUE));
		bean.setUpdateDate(rs.getTimestamp(UPDATE_DATE));
		
		return bean;
	}
}
