/* $Id: DerbyApplicationTypeDAO.java,v 1.3 2007/11/27 02:06:21 knakata Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.ApplicationTypeBean;
import jp.co.argo21.nautica.workflow.dataaccess.ApplicationTypeDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * ApplicationTypeBeanをデータとするDerby用DAOクラスである。
 * <pre><tt>
 * WF_APPTYPEテーブル
 * +---------------+-------------+------+-----+---------+-------+
 * | Field         | Type        | Null | Key | Default | Extra |
 * +---------------+-------------+------+-----+---------+-------+
 * | APP_TYPE      | int(3)      |      | PRI | 0       |       |
 * | APP_TYPE_NAME | varchar(64) |      |     |         |       |
 * +---------------+-------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  knakata(Argo 21, Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class DerbyApplicationTypeDAO extends AbstractDAO implements ApplicationTypeDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** アプリケーションタイプのカラム名 */
	public static final String APP_TYPE      = "APP_TYPE";
	/** アプリケーションタイプ名のカラム名 */
	public static final String APP_TYPE_NAME = "APP_TYPE_NAME";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from WF_APPTYPE where APP_TYPE = ? ";

	private static final String SQL_INSERT
	= "insert into WF_APPTYPE ("
		+ APP_TYPE + ", "
		+ APP_TYPE_NAME+ ") values (?,?)";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	DerbyApplicationTypeDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * 主キーにより、ApplicationTypeBeanを取得する。
	 *
	 * @param appType アプリケーションタイプ
	 * @return ApplicationTypeBean
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ApplicationTypeDAO#findByPrimaryKey(java.lang.String)
	 */
	public ApplicationTypeBean findByPrimaryKey(String appType)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, appType);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ApplicationTypeBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * ApplicationTypeBeanを挿入する。
	 *
	 * @param bean ApplicationTypeBean
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ApplicationTypeDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.ApplicationTypeBean)
	 */
	public void insert(ApplicationTypeBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setInt(1, bean.getAppType());
			st.setString(2, bean.getAppTypeName());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// アプリケーション型レコードの挿入に失敗しました。
				String errMsg = StringManager.get("E0013");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットからApplicationTypeBeanを生成する。
	 *
	 * @param rs 結果セット
	 * @return ApplicationTypeBean
	 * @throws SQLException 任意のSQL例外
	 */
	private ApplicationTypeBean createBean(ResultSet rs)
	throws SQLException
	{
		ApplicationTypeBean bean = new ApplicationTypeBean();

		bean.setAppType(rs.getInt(APP_TYPE));
		bean.setAppTypeName(rs.getString(APP_TYPE_NAME));
		
		return bean;
	}
}
