/* $Id: VariableDAO.java,v 1.3 2007/12/05 05:56:17 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess;

import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * VariableBeanをデータとするDAOインタフェース。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public interface VariableDAO
{
	/**
	 * プロセスIDとプロセス変数IDでプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param vid プロセス変数ID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public VariableBean findByPrimaryKey(String pid, String vid)
	throws SQLException, DAOException;

	/**
	 * プロセスIDとプロセス変数定義IDでプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param vdid プロセス変数定義ID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public VariableBean findByDefinitionID(String pid, String vdid)
	throws SQLException, DAOException;

	/**
	 * プロセスIDとプロセス変数名でプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param name プロセス変数名
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public VariableBean findByName(String pid, String name)
	throws SQLException, DAOException;

	/**
	 * プロセスIDとプロセス変数IDでプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public VariableBean[] findByProcessID(String pid)
	throws SQLException, DAOException;

	/**
	 * リストとしてプロセスIDを収録する。
	 *
	 * @param pid プロセスID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public List<VariableBean> findByProcessIDAsList(String pid)
	throws SQLException, DAOException;
	
	/**
	 * プロセスIDとSQL部分文字列でプロセス変数を取得する。
	 *
	 * @param pid プロセスID
	 * @param sqlPart SQL部分文字列
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public VariableBean[] findBySQLPart(String pid, String sqlPart)
	throws SQLException, DAOException;

	/**
	 * フィルターで特定されたプロセス変数を返す。
	 *
	 * @param filter フィルター
	 * @param pid プロセスID
	 * @return プロセス変数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public VariableBean[] findByFilter(Filter filter, String pid)
	throws DAOException, SQLException, InvalidFilterException;

	/**
	 * プロセス変数を挿入する。
	 *
	 * @param bean プロセス変数
	 * @param aid アクティビティID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void insert(VariableBean bean, String aid)
	throws SQLException, DAOException;
	
	/**
	 * プロセス変数を更新する。
	 *
	 * @param bean プロセス変数
	 * @param value 値の文字列表現
	 * @param aid アクティビティID
	 * @param ts タイムスタンプ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void update(VariableBean bean, String value, String aid, Timestamp ts)
	throws SQLException, DAOException;
}
