/* $Id: ProcessDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess;

import java.sql.SQLException;
import java.sql.Timestamp;

import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;

/**
 * ProcessBeanをデータとするDAOインタフェース。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface ProcessDAO
{
	/**
	 * 主キーにより、プロセスを取得する。
	 *
	 * @param pid プロセスID
	 * @return プロセス
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public ProcessBean findByPrimaryKey(String pid)
	throws SQLException, DAOException;

	/**
	 * プロセス定義IDとプロセスIDからプロセスを取得する。
	 *
	 * @param pdid	プロセス定義ID
	 * @param pid プロセスID
	 * @return プロセス
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public ProcessBean findByPDIDandPID(String pdid, String pid)
	throws SQLException, DAOException;
	
	/**
	 * プロセス定義IDに一致するプロセスを取得する。
	 *
	 * @param pdid	プロセス定義ID
	 * @return プロセス
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public ProcessBean[] findByPDID(String pdid)
	throws SQLException, DAOException;
		
	/**
	 * プロセス定義IDに一致するプロセスの個数を返す。
	 *
	 * @param pdid	プロセス定義ID
	 * @return プロセス個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public int findCountByPDID(String pdid)
	throws SQLException, DAOException;

	/**
	 * フィルターに一致したプロセスを返す。
	 *
	 * @param filter フィルター
	 * @return プロセス
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException
	 */
	public ProcessBean[] findByFilter(Filter filter)
	throws SQLException, DAOException, InvalidFilterException;

	/**
	 * プロセス定義IDとフィルターに一致したプロセスを返す。
	 *
	 * @param pdid	プロセス定義ID
	 * @param filter フィルター
	 * @return プロセス
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException
	 */
	public ProcessBean[] findByFilter(String pdid, Filter filter)
	throws SQLException, DAOException, InvalidFilterException;
	
	/**
	 * プロセスを挿入する。
	 *
	 * @param bean プロセス
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void insert(ProcessBean bean)
	throws SQLException, DAOException;

	/**
	 * プロセス名を更新する。
	 *
	 * @param bean プロセス
	 * @param name プロセス名
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateName(ProcessBean bean, String name)
	throws SQLException, DAOException;
	
	/**
	 * プロセス状態を更新する。
	 *
	 * @param bean プロセス
	 * @param state プロセス状態
	 * @param ts タイムスタンプ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateState(ProcessBean bean, ProcessState state, Timestamp ts)
	throws SQLException, DAOException;

	/**
	 * プロセス変更日付を更新する。
	 *
	 * @param bean プロセス
	 * @param ts タイムスタンプ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateDate(ProcessBean bean, Timestamp ts)
	throws SQLException, DAOException;
}
