/* $Id: ActivityDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess;

import java.sql.SQLException;
import java.sql.Timestamp;

import jp.co.argo21.nautica.workflow.wfmc.ActivityState;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * ActivityBeanをデータとするDAOインタフェース。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface ActivityDAO
{
	/**
	 * 主キーにより、ActivityBeanを取得する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @return アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public ActivityBean findByPrimaryKey(String pid, String aid)
	throws SQLException, DAOException;

	/**
	 * プロセス内で指定されたブロックIDを持ち、実行中のアクティビティの個数を返す。
	 *
	 * @param pid プロセスID
	 * @param bid ブロックID
	 * @return アクティビティの個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public int findOpeningActivityCountByBlockID(String pid, String bid)
	throws SQLException, DAOException;

	/**
	 * プロセス内のアクティビティの個数を返す。
	 *
	 * @param pid プロセスID
	 * @return アクティビティの個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public int findOpeningActivityCountByProcessID(String pid)
	throws SQLException, DAOException;

	/**
	 * プロセス内のアクティビティを返す。
	 *
	 * @param pid プロセスID
	 * @return アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public ActivityBean[] findOpeningActivityByProcessID(String pid)
	throws SQLException, DAOException;

	/**
	 * ブロックIDに対応するブロックの並行処理終了アクティビティを取得する。
	 *
	 * @param pid プロセスID
	 * @param adid アクティビティ定義ID
	 * @param bid ブロックID
	 * @return アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public ActivityBean findConcurrentEnd(String pid, String adid, String bid)
	throws SQLException, DAOException;

	/**
	 * 指定されたフィルターで特定されるアクティビティを返す。
	 *
	 * @param filter フィルター
	 * @return アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 無効なフィルターが指定された場合
	 */
	public ActivityBean[] findByFilter(Filter filter)
	throws SQLException, DAOException, InvalidFilterException;

	/**
	 * 指定されたプロセスとフィルターで特定されるアクティビティを返す。
	 *
	 * @param pid プロセスID
	 * @param filter フィルター
	 * @return アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 無効なフィルターが指定された場合
	 */
	public ActivityBean[] findByFilter(String pid, Filter filter)
	throws SQLException, DAOException, InvalidFilterException;

	/**
	 * 指定された定義IDとフィルターで特定されるアクティビティを返す。
	 *
	 * @param pdid プロセス定義ID
	 * @param adid アクティビティ定義ID
	 * @param filter フィルター
	 * @return アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 無効なフィルターが指定された場合
	 */
	public ActivityBean[] findByFilter(String pdid, String adid, Filter filter)
	throws SQLException, DAOException, InvalidFilterException;
	
	/**
	 * 指定されたアクティビティを挿入する。
	 *
	 * @param bean アクティビティ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void insert(ActivityBean bean)
	throws SQLException, DAOException;
	
	/**
	 * アクティビティの状態を更新する
	 *
	 * @param bean アクティビティ
	 * @param state アクティビティ状態
	 * @param ts タイムスタンプ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateState(ActivityBean bean, ActivityState state, Timestamp ts)
	throws SQLException, DAOException;
	
	/**
	 * 更新日付を更新する。
	 *
	 * @param bean アクティビティ
	 * @param ts タイムスタンプ
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateDate(ActivityBean bean, Timestamp ts)
	throws SQLException, DAOException;
	
	/**
	 * ブロックIDを更新する。
	 *
	 * @param bean アクティビティ
	 * @param bid ブロックID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateBlockID(ActivityBean bean, String bid)
	throws SQLException, DAOException;
}
