/* $Id: WorkflowServiceAccessor.java,v 1.1 2007/10/29 02:55:41 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.client;

import javax.naming.NamingException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.util.GeronimoUtils;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.ConnectionFailedException;
import jp.co.argo21.nautica.workflow.wfmc.DefinitionRepository;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowAdminHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowEngineHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowServiceManager;

/**
 * ワークフローサービスアクセッサは、ワークフローエンジンに対する
 * クライアントライブラリである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class WorkflowServiceAccessor
{
	private static final String WORKFLOW_SERVICE_MANAGER_REMOTE_IF =
		"nautica-workflow/WorkflowServiceManagerBean/jp.co.argo21.nautica.workflow.wfmc.WorkflowServiceManager";
	
	/** ログ出力オブジェクト */
    private Logger log = Logger.getLogger(this.getClass());
    
    /** ワークフローエンジン接続情報 */
    private ConnectionInfo connInfo;
    /** WorkflowServiceManagerのリモートオブジェクト */
    private WorkflowServiceManager wsm;
    /** セッションID */
    private String session;
    
	/**
	 * ワークフローサービスアクセッサを生成する。
	 *
	 * @param conn 接続情報
	 */
	public WorkflowServiceAccessor(ConnectionInfo conn)
	{
	    if (conn == null) {
	        throw new IllegalArgumentException();
	    }
	    
	    this.connInfo = conn;
		
	    // ログ出力
	    log.info(StringManager.get("I8001"));
	    log.info("ConnectionInfomation(" + conn.toString() + ")");
	}
	
	/**
	 * 現在のアクセッサが保持している情報でセッションを確立し、
	 * セッションIDを返す。
	 * ワークフローサービスマネージャに対して、認証を行う。
	 * 認証に成功した場合は、サービスマネージャの参照を保持する。
	 * close()が呼ばれていない場合は、まずclose()を行ってから、
	 * 接続を開きなおす。
	 * その場合、以前のセッションIDは無効となる。
	 *
	 * @return セッションID
	 * @throws ConnectionFailedException 認証失敗の場合
	 */
	public String open() 
	throws ConnectionFailedException
	{
	    // 既に接続済みかどうかチェックする
	    // 接続済みの場合、セッションIDが保存されているので、
	    // セッションIDがnull以外の場合、既に接続済みである
	    // 接続済みで、再度open()メソッドを呼ぶ場合、close()してから接続を開きなおす
	    if (getSession() != null) {
	        // ログ出力
	        log.info(StringManager.get("W8001"));
	        try {
	            this.close();
	        } catch (InvalidSessionException ex) {
	            /* セッションが無効な場合の例外は無視する */
	        }
	    }
	    
	    // 接続情報を元に、WorkflowServiceManagerのリモートオブジェクトを取得する
	    try {
	    	this.wsm = (WorkflowServiceManager)GeronimoUtils.getEJB(
	    			this.connInfo.getEngineName(),
	    			WORKFLOW_SERVICE_MANAGER_REMOTE_IF);
	    } catch (NamingException ex) {
	        // ネーミング例外が発生した場合
	        // ログ出力：ワークフローエンジンと接続できませんでした。
	        log.error(StringManager.get("E8001"));
	        // 例外メッセージ：ワークフローエンジンと接続できませんでした。
	        throw new ConnectionFailedException(StringManager.get("E8001"), ex);
	    }
	    
	    // WorkflowServiceManager の connect()メソッドを使用して、認証を行う。
	    this.session = this.wsm.connect(this.connInfo.getUserID(), 
	                                      this.connInfo.getPassword());

	    // ログ出力：ワークフローエンジンと接続しました。
	    log.info(StringManager.get("I8002"));
	    
	    return getSession();
	}
	
	/**
	 * セッションを明示的に閉じる。ただし、セッションはサーバ側で
	 * タイムアウトによって無効になっていることがある。
	 * その場合は、InvalidSessionExceptionが返るが、
	 * 無視してもかまわない。
	 *
	 * @throws InvalidSessionException セッションが無効の場合
	 */
	public void close()
	throws InvalidSessionException
	{
	    try {
	        // ワークフローエンジンから切断する
	        this.wsm.disconnect(getSession());
	    } finally {
	        // セッションIDをクリアする
	        this.session = null;
	        
	        // WorkflowServiceManager のリモートオブジェクトをクリアする
	        this.wsm = null;
	        
	        // ログ出力：ワークフローエンジンから切断しました。
	        log.info(StringManager.get("I8003"));
	    }
	}

	/**
	 * 定義リポジトリを返す。
	 *
	 * @return 定義リポジトリ 
	 * @throws InvalidSessionException セッションが無効の場合
	 */
	public DefinitionRepository getDefinitionRepository()
	throws InvalidSessionException
	{
		
		return this.wsm.getDefinitionRepository(getSession());
	}

	/**
	 * 作業項目ハンドラを返す。
	 *
	 * @return 作業項目ハンドラを返す。
	 * @throws InvalidSessionException セッションが無効の場合
	 */
	public WorkItemHandler getWorkItemHandler()
	throws InvalidSessionException
	{
		return this.wsm.getWorkItemHandler(getSession());
	}

	/**
	 * ワークフローエンジンハンドラを返す。
	 *
	 * @return ワークフローエンジンハンドラ
	 * @throws InvalidSessionException セッションが無効の場合
	 */
	public WorkflowEngineHandler getWorkflowEngineHandler()
	throws InvalidSessionException
	{
		return this.wsm.getWorkflowEngineHandler(getSession());
	}

	/**
	 * ワークフロー管理ハンドラを返す。
	 *
	 * @return ワークフロー管理ハンドラ
	 * @throws InvalidSessionException セッションが無効の場合
	 */
	public WorkflowAdminHandler getWorkflowAdminHandler()
	throws InvalidSessionException
	{
		return this.wsm.getWorkflowAdminHandler(getSession());
	}
	
	/**
	 * セッションIDを取得する
	 * 
	 * @return	セッションID。未設定の場合は<code>null</code>が返る。
	 */
	private String getSession()
	{
	    return this.session;
	}
}
