/* $Id: InterWorkflowConnector.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.wfmc;

import jp.co.argo21.nautica.workflow.omg.WorkflowException;

/**
 * エンジン間接続のためのインタフェースとして、InterWorkflowConnectorを提供する。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public interface InterWorkflowConnector
{
	/** 結果の返却を必要としない */
	public static final int NOT_NEED_RESULT = 0;
	/** 結果の返却を必要とする */
	public static final int NEED_RESULT     = 1;

	/**
	 * 自エンジンのIDを指定して、リモートエンジンアクセス用のIDを返す。
	 *
	 * @param seid 自エンジンのID
	 * @param password パスワード
	 * @return リモートのエンジンから返されるエンジン接続ID
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String connect(String seid, String password)
	throws WorkflowException;

	/**
	 * リモートのエンジンとの接続を切断する。
	 *
	 * @param eid エンジンID
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void disconnect(String eid)
	throws WorkflowException;

	/**
	 * プロセスインスタンス状態の変更を要求する。
	 *
	 * @param eid エンジンID
	 * @param pid プロセスインスタンスID
	 * @param state プロセスインスタンス状態
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void changeProcessState(String eid, String pid, ProcessState state)
	throws WorkflowException;

	/**
	 * プロセスインスタンスの生成を要求する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param pdid プロセス定義ID
	 * @param flag 戻り値を返す必要があるかどうかを表すフラグ
	 * @param rootpid 要求元のプロセスインスタンスID
	 * @param aid 要求元のアクティビティインスタンスID
	 * @param userid ユーザID。主に記録用。
	 * @param roleid 権限ID。主に記録用。
	 * @return 生成されたプロセスインスタンスのID
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String createProcess(String eid, String pdid, int flag,
			String rootpid, String aid, String userid, String roleid)
	throws WorkflowException;

	/**
	 * プロセスインスタンス属性の取得を要求する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param pid プロセスインスタンスID
	 * @param rootpid 要求元のプロセスインスタンスID
	 * @param aid アクティビティインスタンスID
	 * @return 属性の配列を返す。
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public Attribute[] getProcessAttributes(String eid, String pid,
			String rootpid, String aid)
	throws WorkflowException;

	/**
	 * プロセスインスタンス状態の取得を要求する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param pid プロセスインスタンスID
	 * @return プロセスインスタンス状態
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public ProcessState getProcessState(String eid, String pid)
	throws WorkflowException;

	/**
	 * プロセスインスタンス属性の変更を通知する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param pid プロセスインスタンスID
	 * @param attr 属性
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void processAttributesChanged(String eid, String pid,
			Attribute[] attr)
	throws WorkflowException;

	/**
	 * プロセスインスタンス状態の変更を通知する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param pid プロセスインスタンスID
	 * @param state プロセスインスタンス状態
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void processStateChanged(String eid,
			String pid, ProcessState state)
	throws WorkflowException;

	/**
	 * プロセスインスタンス属性の設定を要求する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param rootpid 要求元のプロセスインスタンスID
	 * @param aid 要求アクティビティインスタンスID
	 * @param pid プロセスインスタンスID
	 * @param attr 属性
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void setProcessAttributes(String eid, String rootpid,
			String aid, String pid, Attribute[] attr)
	throws WorkflowException;

	/**
	 * トリガーとなるアクティビティを通知する。
	 * これは、起動したアクティビティとは別なアクティビティによって
	 * プロセスを制御する場合の通知として使用する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param rootpid 要求元のプロセスインスタンスID
	 * @param aid 要求アクティビティインスタンスID
	 * @param pid プロセスインスタンスID
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void triggerActivity(String eid, String rootpid,
			String aid, String pid)
	throws WorkflowException;

	/**
	 * このエンジンから要求されて動作中のプロセスインスタンスID一覧の
	 * 取得を要求する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param filter 検索用フィルター
	 * @return プロセスインスタンスID一覧
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public String[] listProcesses(String eid, Filter filter)
	throws WorkflowException;

	/**
	 * プロセスインスタンスの制御の放棄を通知する。
	 *
	 * @param eid リモートエンジン接続ID
	 * @param pid プロセスインスタンスID
	 * @throws WorkflowException　ワークフロー関連で例外が発生した場合
	 */
	public void relinquishProcess(String eid, String pid)
	throws WorkflowException;
}
