/* $Id: FilterHelper.java,v 1.1 2007/10/29 02:55:37 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.util;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.filter.GroupFilter;
import jp.co.argo21.nautica.workflow.filter.StateFilter;
import jp.co.argo21.nautica.workflow.omg.WfState;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemState;

/**
 * フィルターに関するヘルパークラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class FilterHelper
{
	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	
	/**
	 * フィルターを指定して、作業項目状態を取り出す。
	 * 引数には、状態フィルターか状態フィルターを収めたグループフィルターを
	 * 指定すること。
	 *
	 * @param filter 状態フィルターか状態フィルターを収めたグループフィルター
	 * @return 作業項目状態
	 * @throws InvalidFilterException 使用できないフィルターが指定されている場合
	 */
	public static WorkItemState[] getWorkItemState(Filter filter)
	throws InvalidFilterException
	{
		List<WfState> list = new ArrayList<WfState>();
		
		parseWorkItemState(list, filter);
		
		return (WorkItemState[])list.toArray(new WorkItemState[list.size()]);
	}
	
	/**
	 * フィルターから適切な状態を取り出す。
	 *
	 * @param list 作業項目状態
	 * @param filter 状態フィルターか状態フィルターを収めたグループフィルター
	 * @throws InvalidFilterException 使用できないフィルターが指定されている場合
	 */
	private static void parseWorkItemState(List<WfState> list, Filter filter)
	throws InvalidFilterException
	{
		if (filter == null) {
			//NOP
		} else if (filter instanceof StateFilter) {
			parseWorkItemState(list, (StateFilter)filter);
		} else if (filter instanceof GroupFilter) {
			parseWorkItemState(list, (GroupFilter)filter);
		} else {
			// 作業項目には使用できないフィルターが指定されています。
			String errMsg = StringManager.get("E0037");
			eLog.error(errMsg);
			throw new InvalidFilterException(errMsg);
		}
	}
	
	/**
	 * フィルターから適切な状態を取り出す。
	 *
	 * @param list 作業項目状態
	 * @param filter 状態フィルター
	 * @throws InvalidFilterException 使用できないフィルターが指定されている場合
	 */
	private static void parseWorkItemState(List<WfState> list, StateFilter filter)
	throws InvalidFilterException
	{
		WfState state = filter.getState();
		if (state instanceof WorkItemState) {
			list.add(state);
		} else {
			// 作業項目状態には使用できないフィルターが指定されています。
			String errMsg = StringManager.get("E0038");
			eLog.error(errMsg);
			throw new InvalidFilterException(errMsg);
		}
	}
	
	/**
	 * フィルターから適切な状態を取り出す。
	 *
	 * @param list 作業項目状態
	 * @param filter 状態フィルターを収めたグループフィルター
	 * @throws InvalidFilterException 使用できないフィルターが指定されている場合
	 */
	private static void parseWorkItemState(List<WfState> list, GroupFilter filter)
	throws InvalidFilterException
	{
		Filter[] children = filter.getChild();
		
		for (int i = 0; i < children.length; i++) {
			parseWorkItemState(list, children[i]);
		}
	}
}
