/* $Id: SystemCommandToolAgent.java,v 1.4 2007/11/28 07:09:06 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ta;

import java.sql.Timestamp;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionBean;
import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.SystemChecker;
import jp.co.argo21.nautica.workflow.engine.WorkflowEngine;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationBusyException;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationNotDefinedException;
import jp.co.argo21.nautica.workflow.wfmc.ApplicationNotStoppedException;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.InvalidToolAgentException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemState;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowServiceManager;

/**
 * システムコマンド用ツールエージェントのクラス。
 * 
 * アプリケーション名で設定ファイルに登録されたコマンドを実行する。<br>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.4 $
 * @since   Nautica Workflow 1.0
 */
public class SystemCommandToolAgent extends AbstractToolAgent {

    /* 実行中・実行済みプロセス */
    private static Map<Long, Process> processes = Collections.synchronizedMap(new HashMap<Long, Process>());
	/** ローカルスクリプト配置パス */
	public static final String LOCAL_COMMAND_PATH;

	static
	{
		String separator = SystemChecker.getProperty("file.separator");
		LOCAL_COMMAND_PATH = TOOLAGENT_PATH + separator + "commands";
	}

    /**
     * コンストラクタ
     *
     * @param userId	ユーザID
     * @param passwd	パスワード
     */
    SystemCommandToolAgent(String userId, String passwd) {
        this.userId = userId;
        this.passwd = passwd;
    }

    /**
     * プロセスを設定する。
     * 
     * 実行中のプロセスを設定する。
     * このことにより、アプリケーションの停止を行うことが出来るようになる。
     *
     * @param handle ハンドル番号
     * @param process プロセスインスタンス
     */
    public static void setProcess(long handle, Process process)
    {
        processes.put(new Long(handle), process);
    }
    
    /**
     * プロセスを返す。
     * 
     * @param	handle	ツールエージェントハンドル
     * @return	<code>Process</code>クラスのインスタンス。
     * 			プロセスが存在しない場合は、<code>null</code>が返る。
     */
    protected static Process getProcess(long handle) {
        return (Process)processes.get(new Long(handle));
    }
    
    /**
	 * アプリケーションの停止を行う。
	 *
	 * @param toolAgentHandle ツールエージェントハンドル
	 * @param wid 作業項目ID
	 * @throws ApplicationNotStoppedException アプリケーションを停止できなかった場合
	 * @throws InvalidToolAgentException 有効でないツールエージェントハンドルを指定した場合
	 * @throws InvalidWorkItemException 有効でない作業項目IDを指定した場合
	 * @throws ApplicationBusyException アプリケーションがビジーで要求を受けられない場合
     */
    public void terminateApplication(int toolAgentHandle, String wid)
            throws ApplicationNotStoppedException, InvalidToolAgentException,
            InvalidWorkItemException, ApplicationBusyException {
        
        // ログ
        String msg = StringManager.get("I2003") 
        	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
        	+ "WorkItemID=" + wid + ")";
        log.info(msg);
        
        // ツールエージェントハンドルに対応したプロセスオブジェクトを取得する
        Process process = getProcess(toolAgentHandle);
        
        // ツールエージェントハンドルに対応したプロセスオブジェクトが存在しない場合、
        // アプリケーションを停止することは出来ないため、例外を投げる
        if (process == null) {
            // ログ
            String errMsg = StringManager.get("E2002")
            	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
            	+ "WorkItemID=" + wid + ")";
            log.error(errMsg);
            throw new InvalidToolAgentException(errMsg);
        }
        
        String session = null;
        try {
            // DAOFactoryを取得
            WorkflowDAOFactory factory = DataAccessManager.getDAOFactory();
            // AppExecutionDAOを取得
            AppExecutionDAO appExecutionDao = factory.getAppExecutionDAO();
            // AppExecutionBeanを取得
            List<AppExecutionBean> beans = appExecutionDao.findByExecID(toolAgentHandle);
        
        	// APPEXECにレコードが1件存在しない場合は、エラー
            if (beans.size() != 1) {
                // エラー
                String errMsg = StringManager.get("E2002")
                	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
                	+ "WorkItemID=" + wid + ")";
                throw new InvalidToolAgentException(errMsg);
            }
           AppExecutionBean bean = beans.get(0);
        
            // アプリケーションを強制停止
            process.destroy();

            // アプリケーションの実行状態を1：異常終了に更新する
            appExecutionDao.updateState(bean.getAppName(), 
                    bean.getToolAgent(), 
                    bean.getExecId(), 
                    bean.getAppState(),
                    ApplicationState.TERMINATED,
                    new Timestamp(System.currentTimeMillis()));
            
            // 作業項目の完了を通知する
            WorkflowServiceManager manager = WorkflowEngine.getInstance();
            session = connect();
            WorkItemHandler handler = manager.getWorkItemHandler(session);
            handler.changeWorkItemState(session, wid, WorkItemState.CLOSED_TERMINATED);
            
        } catch (Exception ex) {
            // ログ
            String errMsg = StringManager.get("E2002")
            	+ "(ToolAgentHandler=" + toolAgentHandle + ", "
            	+ "WorkItemID=" + wid + ")";
            log.error(errMsg, ex);
            throw new InvalidToolAgentException(errMsg);
        } finally {
            try {
                if (session != null) {
                    disconnect(session);
                }
            } catch (Exception ex) {
                /* Ignore */
            }
        }
        
        // ログ
        log.info(StringManager.get("I2004"));
    }
    
    /**
     * アプリケーションを起動するツールエージェントのクラス名を取得する。
     * 
     * @return	アプリケーションを起動するツールエージェントのクラス名
     */
    protected String getToolAgentClassName() {
        return this.getClass().getName();
    }
    
    /**
     * アプリケーション実行情報を送信するキュー名を取得する。
     * 
     * ツールエージェント毎にキューが必要で、そのキューに名前をつけなくてはならない。
     * このキュー名はツールエージェント固有の文字列である。
     * 基本的にはツールエージェント名を使用する。
     * 各ツールエージェントで重複しないように注意すること。
     *
     * @return	キュー名
     */
    protected String getQueueName() {
        return SystemCommandToolAgentFactory.QUEUE_NAME;
    }
    
    /**
     * キューに送信するアプリケーション実行情報を取得する。
     *
     * アプリケーション実行クラスでは、この情報を基にして、
     * アプリケーションを実行する。 
     *
     * @param appName	定義に指定したアプリケーション名
     * @param wid		作業項目ID
     * @param handler	ツールエージェントハンドラ
     * @param attrList	属性一覧
     * @return	アプリケーション実行情報
     */
    protected AppExecutionInfo getExecInfo(String appName, String wid,
            int handler, Attribute[] attrList) 
    {
        return new SystemCommandExecutionInfo(appName, getToolAgentClassName(), 
                wid, handler, attrList);
    }
    
    /**
     * アプリケーションが定義されているかどうかチェックする。
     * 
     * アプリケーションが定義されていない場合、例外が発生する。
     *
     * @param appName	アプリケーション名
     * @throws	ApplicationNotDefinedException	アプリケーションが定義されていない場合
     * @see jp.co.argo21.nautica.workflow.ta.AbstractToolAgent#checkApplicationDefined(java.lang.String)
     */
    protected void checkApplicationDefined(String appName)
    	throws ApplicationNotDefinedException {
        
        SystemCommandConfig config = null;
        try {
	        config = SystemCommandConfigFactory.getConfig();
        } catch (Exception ex) {
            // エラー
            String errMsg = StringManager.get("E2004")
            	+ "(ApplicationName=" + appName + ")";
            throw new ApplicationNotDefinedException(errMsg, ex);
        }
        
        if (config == null || !config.existApplication(appName)) {
            // エラー
            String errMsg = StringManager.get("E2004")
        		+ "(ApplicationName=" + appName + ")";
            throw new ApplicationNotDefinedException(errMsg);
        }
    }
}
