/* $Id: SimpleRoleManager.java,v 1.1 2007/10/29 02:55:37 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.security.simple;

import jp.co.argo21.nautica.workflow.dataaccess.RoleManagerDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.security.Organization;
import jp.co.argo21.nautica.workflow.security.Role;
import jp.co.argo21.nautica.workflow.security.RoleManager;
import jp.co.argo21.nautica.workflow.security.User;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowSecurityException;

import org.apache.log4j.Logger;

/**
 * 本ワークフローシステム運用基盤および外部から利用される権限管理関連の各種APIを
 * 提供する参考実装である。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class SimpleRoleManager implements RoleManager {

	private static Logger sLog = LogManager.getSercurityLogger();
	
	/**
	 * SimpleRoleManager を生成する。
	 * 
	 * @throws Exception 組織管理の公開に失敗した場合
	 */
	public SimpleRoleManager() throws Exception {
		try {
			sLog.info(StringManager.get("I1007"));
		} catch (Exception ex) {
			// 権限管理の公開に失敗しました。
			String errMsg = StringManager.get("E1009");
			sLog.error(errMsg, ex);
			throw ex;
		
		}
	}
	
	/**
	 * ridに対応する権限を返す。
	 * ridに対応する権限がない場合は null を返す。
	 *
	 * @param  rid 権限ID
	 * @return 権限
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Role getRole(String rid)
	throws WorkflowSecurityException {
	   
	    try {
	        RoleManagerDAO rolemanagerDAO = getRoleManagerDAO();	        
	        Role role = rolemanagerDAO.findByRoleID(rid);
	        
	        return role;
	    
	    } catch (Exception e) {
	        // 権限の取得に失敗しました。
	        String errMsg = StringManager.get("E1022") + "(RoleID = " + rid + ")";
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}

	/**
	 * 組織に付与されている全権限を返す。
	 * 指定された組織に付与されている権限が存在しない場合、および
	 * 指定された組織が存在しない場合は、空の配列を返す。
	 *
	 * @param  org 組織単位
	 * @return 権限の一覧
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Role[] getAttachedRoles(Organization org)
	throws WorkflowSecurityException {
		
	    try {	        
	        RoleManagerDAO rolemanagerDAO = getRoleManagerDAO();
	        Role[] roles = rolemanagerDAO.findByOrganizationID(org.getID());
	        
	        return roles;
	    
	    } catch (Exception e) {
	        // 割り当て権限の取得に失敗しました。
	        String errMsg = StringManager.get("E1023");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}

	/**
	 * ユーザに付与されている全権限を返す。
	 * 指定されたユーザに付与されている権限が存在しない場合、および
	 * 指定されたユーザが存在しない場合は、空の配列を返す。
	 *
	 * @param  user ユーザ
	 * @return 権限の一覧
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Role[] getAttachedRoles(User user)
	throws WorkflowSecurityException {
		
	    try {	        
	        RoleManagerDAO rolemanagerDAO = getRoleManagerDAO();
	        Role[] roles = rolemanagerDAO.findByUserID(user.getID());
	        
	        return roles;
	        	        
	    } catch (Exception e) {
	        // 割り当て権限の取得に失敗しました。
	        String errMsg = StringManager.get("E1023");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}

	/**
	 * 権限を持つ全組織単位を返す。
	 * 指定された権限を持つ組織が存在しない場合、および
	 * 指定された権限が存在しない場合は、空の配列を返す。
	 *
	 * @param  role 権限
	 * @return 組織単位の一覧
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public Organization[] getOrganizationsByRole(Role role)
	throws WorkflowSecurityException {

	    try {
	        RoleManagerDAO rolemanagerDAO = getRoleManagerDAO();
	        Organization[] orgs = rolemanagerDAO.findOrganizationsByRoleID(role.getID());
	        
	        return orgs;
	        
	    } catch (Exception e) {
	        // 所属組織の取得に失敗しました。
	        String errMsg = StringManager.get("E1020");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}

	/**
	 * 権限を持つ全ユーザを返す。
	 * 指定された権限を持つユーザが存在しない場合、および
	 * 指定された権限が存在しない場合は、空の配列を返す。
	 *
	 * @param  role 権限
	 * @return ユーザの一覧
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	public User[] getUsersByRole(Role role)
	throws WorkflowSecurityException {
		
	    try {	        
	        RoleManagerDAO rolemanagerDAO = getRoleManagerDAO();
	        User[] users = rolemanagerDAO.findUsersByRoleID(role.getID());
	        
	        return users;
	        
	    } catch (Exception e) {
	        // 所属ユーザの取得に失敗しました。
	        String errMsg = StringManager.get("E1019");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}
	
	/**
	 * RoleManagerDAOオブジェクトを返す。
	 * 
	 * @return RoleManegerDAOオブジェクト
	 * @throws Exception データベースアクセスエラーが発生した場合
	 */
	private RoleManagerDAO getRoleManagerDAO() throws Exception {
	    WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
        return daoFactory.getRoleManagerDAO();
	}
}