/* $Id: SimpleOrganizationManager.java,v 1.1 2007/10/29 02:55:37 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.security.simple;

import jp.co.argo21.nautica.workflow.dataaccess.OrganizationManagerDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.engine.DataAccessManager;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.security.InvalidOrganizationException;
import jp.co.argo21.nautica.workflow.security.Organization;
import jp.co.argo21.nautica.workflow.security.OrganizationManager;
import jp.co.argo21.nautica.workflow.security.User;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowSecurityException;

import org.apache.log4j.Logger;

/**
 * 本ワークフローシステム運用基盤および外部から利用される組織管理関連の各種APIを
 * 提供する参考実装である。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class SimpleOrganizationManager implements OrganizationManager {

	private static Logger sLog = LogManager.getSercurityLogger(); 
	
	/**
	 * SimpleOrganizationManager を生成する。
	 * 
	 * @throws Exception 組織管理の公開に失敗した場合
	 */
	public SimpleOrganizationManager() throws Exception {
		try {
			sLog.info(StringManager.get("I1006"));
		} catch (Exception ex) {
			// 組織管理の公開に失敗しました。
			String errMsg = StringManager.get("E1007");
			sLog.error(errMsg, ex);
			throw ex;
		}
	}
	
	/**
	 * oidに対応する組織を返す。
	 * oidに対応する組織が存在しない場合は null を返す。
	 *
	 * @param  oid 組織ID
	 * @return 組織単位
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 * @see jp.co.argo21.nautica.workflow.security.OrganizationManager#getOrganization(java.lang.String)
	 */
	public Organization getOrganization(String oid)
	throws WorkflowSecurityException {
		
	    try {	        
	        OrganizationManagerDAO orgmanagerDAO = getOrganizationManagerDAO();
	        Organization org = orgmanagerDAO.findByOrganizationID(oid);
	        
	        return org;
	        
	    } catch (Exception e) {
	        // 組織情報の取得に失敗しました。
	        String errMsg = StringManager.get("E1018") + "(OrganizationID = " + oid + ")";
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg , e);
	    }
	}

	/**
	 * 組織に属しているユーザの一覧を返す。
	 * 指定された組織に属するユーザが存在しない場合、および
	 * 指定された組織が存在しない場合は、空の配列を返す。
	 *
	 * @param  org 組織単位
	 * @return ユーザ一覧
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 * @see jp.co.argo21.nautica.workflow.security.OrganizationManager#getAssignedUser(jp.co.argo21.nautica.workflow.security.Organization)
	 */
	public User[] getAssignedUser(Organization org)
	throws WorkflowSecurityException {
		
	    try {
	        OrganizationManagerDAO orgmanagerDAO = getOrganizationManagerDAO();
	        User[] users = orgmanagerDAO.findUsersByOrganizationID(org.getID());
	        
	        return users;
	        
	    } catch (Exception e) {
	        // 所属ユーザの取得に失敗しました。
	        String errMsg = StringManager.get("E1019");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}

	/**
	 * ユーザの所属する組織を返す。
	 * 指定されたユーザの所属する組織が存在しない場合、および
	 * 指定されたユーザが存在しない場合は、空の配列を返す。
	 * 
	 * @param  user ユーザ
	 * @return 組織一覧
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 * @see jp.co.argo21.nautica.workflow.security.OrganizationManager#getOrganizationByUser(jp.co.argo21.nautica.workflow.security.User)
	 */
	public Organization[] getOrganizationByUser(User user)
	throws WorkflowSecurityException {
		
	    try {
	        OrganizationManagerDAO orgmanagerDAO = getOrganizationManagerDAO();
	        Organization[] orgs = orgmanagerDAO.findByUserID(user.getID());
	        
	        return orgs;
	        
	    } catch (Exception e) {
	        // 所属組織の取得に失敗しました。
	        String errMsg = StringManager.get("E1020");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}
	
	/**
	 * 該当組織の上位組織を返す。
	 * 上位組織が存在しない場合は null を返す。
	 *
	 * @param  org 組織単位
	 * @return 上位組織
	 * @throws InvalidOrganizationException 指定された組織が有効でない場合
	 * @throws WorkflowSecurityException    その他のセキュリティ例外
	 * @see jp.co.argo21.nautica.workflow.security.simple.OrganizationManager#getHigherOrganization(jp.co.argo21.nautica.workflow.security.Organization)
	 */
	public Organization getHigherOrganization(Organization org)
	throws InvalidOrganizationException, WorkflowSecurityException {
	    try {
	        OrganizationManagerDAO orgmanagerDAO = getOrganizationManagerDAO();
	        Organization higher = orgmanagerDAO.findHigherByOrganizationID(org.getID());
	        
	        return higher;
	        
	    } catch (Exception e) {
	        // 上位組織の取得に失敗しました。
	        String errMsg = StringManager.get("E1021");
	        sLog.error(errMsg);
	        throw new WorkflowSecurityException(errMsg, e);
	    }
	}
	
	/**
	 * OrganizationManagerDAOオブジェクトを返す。
	 * 
	 * @return OrganizationManagerDAOオブジェクト
	 * @throws Exception データベースアクセスエラーが発生した場合
	 */
	private OrganizationManagerDAO getOrganizationManagerDAO() throws Exception {
	    WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
	    return daoFactory.getOrganizationManagerDAO();
	}
}
