/* $Id: ConditionalExpression.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.expression;

import java.util.List;

import jp.co.argo21.nautica.workflow.xpdl.ConditionalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.LogicalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.RelationalExpressionDocument;
import jp.co.argo21.nautica.workflow.xpdl.UnaryExpressionDocument;

import org.apache.xmlbeans.XmlObject;

/**
 * 遷移条件式
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ConditionalExpression implements Expression
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** 下位の条件式 */
	private Expression subExpression;
	
	/**
	 * 遷移条件式
	 * 
	 * @param exp	条件式
	 */
	public ConditionalExpression(Expression exp)
	{
		subExpression = exp;
	}

	/**
	 * XMLから遷移条件式を生成する。
	 * 
	 * @param exp	XML
	 */
	public ConditionalExpression(ConditionalExpressionDocument.ConditionalExpression exp)
	{
		if (exp.isSetUnaryExpression()) {
			UnaryExpressionDocument.UnaryExpression unary = exp.getUnaryExpression();
			subExpression = new UnaryExpression(unary);
		} else if (exp.isSetLogicalExpression()) {
			LogicalExpressionDocument.LogicalExpression logic = exp.getLogicalExpression();
			subExpression = new LogicalExpression(logic);
		} else if (exp.isSetRelationalExpression()) {
			RelationalExpressionDocument.RelationalExpression rel = exp.getRelationalExpression();
			subExpression = new RelationalExpression(rel);
		}
	}
	
	/**
	 * 引数のオブジェクトに条件式を格納する。
	 * 
	 * @param o	XMLオブジェクト
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#saveTo(org.apache.xmlbeans.XmlObject)
	 */
	public void saveTo(XmlObject o)
	{
		if (o == null) return;
		if (subExpression == null) return;
		if (!(o instanceof ConditionalExpressionDocument.ConditionalExpression)) return;

		ConditionalExpressionDocument.ConditionalExpression ce
			= (ConditionalExpressionDocument.ConditionalExpression)o;

		XmlObject exp = null;
		if (subExpression instanceof UnaryExpression) {
			exp = ce.addNewUnaryExpression();
		} else if (subExpression instanceof LogicalExpression) {
			exp = ce.addNewLogicalExpression();
		} else if (subExpression instanceof RelationalExpression) {
			exp = ce.addNewRelationalExpression();
		}
		subExpression.saveTo(exp);
	}

	/**
	 * 引数の情報を元に条件を評価する。
	 * 
	 * @param list	変数の現在値
	 * @return	条件が満たされた場合はtrue
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#eval(java.util.List)
	 */
	@SuppressWarnings("unchecked")
	public boolean eval(List list)
	{
		if (subExpression == null) return true;
		else return subExpression.eval(list);
	}

	/**
	 * 条件式の文字列表現を返す。
	 * 
	 * @return	文字列表現
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#getExpressionString()
	 */
	public String getExpressionString()
	{
		String s = "null";
		if (subExpression != null) s = subExpression.getExpressionString();

		return "(" + s + ")";
	}

	/**
	 * 変数一覧を利用しながら条件式の文字列表現を返す。
	 * 
	 * @param vars	変数一覧
	 * @return	文字列表現
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#getExpressionString(java.util.List)
	 */
	@SuppressWarnings("unchecked")
	public String getExpressionString(List vars)
	{
		String s = "null";
		if (subExpression != null) s = subExpression.getExpressionString(vars);

		return "(" + s + ")";
	}

	/**
	 * この条件式に含まれるすべての条件式を返す。
	 * 
	 * @param exps	条件式一覧
	 * @see jp.co.argo21.nautica.workflow.expression.Expression#getAllExpressions(java.util.List)
	 */
	@SuppressWarnings("unchecked")
	public void getAllExpressions(List exps)
	{
		if (subExpression != null) subExpression.getAllExpressions(exps);
	}
}
