/* $Id: WorkItemHandlerImpl.java,v 1.2 2007/11/26 08:51:40 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.security.SessionManager;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.AttributeAssignmentFailedException;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidAttributeException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSourceUserException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidTargetUserException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.Participant;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItem;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemState;

/**
 * 作業項目ハンドラは、作業項目操作に対するクライアントインタフェースの
 * 具体実装である。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class WorkItemHandlerImpl implements WorkItemHandler
{
	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	/** 監査ログ */
	private static Logger audit = LogManager.getAuditLogger();

	/**
	 * WorkItemHandlerImplを生成する。
	 */
	WorkItemHandlerImpl()
	{
	}

	/**
	 * 作業項目一覧の取得を行う。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return 作業項目一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItems(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public WorkItem[] getWorkItems(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getWorkItems,"
			 + "session = [" + session + "],"
			 + "filter = [" + filter + "])";
	
		try {
			DataAccessManager.begin(false);
			
			validateSession(session);

			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			WorkItem[] workItems = container.getWorkItems(filter);
			
			DataAccessManager.commit();

			return workItems;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目一覧の取得に失敗しました。
			String E0174 = StringManager.get("E0174") + argInfo;
			eLog.error(E0174, ex);
			throw new InvalidFilterException(E0174, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}

	/**
	 * 作業項目の取得を行う。
	 * このAPI呼び出しで、該当作業項目がセッションによってロックされる。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @return 作業項目
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItem(java.lang.String, java.lang.String)
	 */
	public WorkItem getWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException
	{
		String argInfo = "(NAUTICA API = getWorkItem,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "])";
	
		try {
			checkWorkItemID(wid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);

			// 監査ログ出力
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMSelectedWorkitem", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			WorkItem workItem = container.getWorkItem(wid);

			data.setDataElement("ActivityInstanceID", workItem.getActivityID());
				
			DataAccessManager.commit();

		    data.setDataElement("WorkItemID", wid);
		    audit.info(data);

			return workItem;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目の取得に失敗しました。
			String E0116 = StringManager.get("E0116") + argInfo;
			eLog.error(E0116, ex);
			throw new InvalidWorkItemException(E0116, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}

	/**
	 * 作業項目の完了を行う。
	 * このAPI呼び出しで、該当作業項目がセッションによってアンロックされる。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#completeWorkItem(java.lang.String, java.lang.String)
	 */
	public void completeWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException
	{
		String argInfo = "(NAUTICA API = completeWorkItem,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "])";
	
		try {
			checkWorkItemID(wid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
			// 監査ログ出力
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMCompletedWorkitem", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);

			data.setDataElement("ActivityInstanceID", container.getWorkItem(wid).getActivityID());

			container.changeWorkItemState(session, wid, WorkItemState.CLOSED_COMPLETED);
				
			DataAccessManager.commit();

		    data.setDataElement("WorkItemID", wid);
		    data.setDataElement("WorkItemState", WorkItemState.CLOSED_COMPLETED);			    
		    audit.info(data);

		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目の完了に失敗しました。
			String E0175 = StringManager.get("E0175") + argInfo;
			eLog.error(E0175, ex);
			throw new InvalidWorkItemException(E0175, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}

	/**
	 * 現在の作業項目が遷移可能な状態の一覧を返す。
	 * 状態がない場合は、空の配列を返す。
	 * 利用可能なフィルターはStateFilterのみとする。
	 * このAPIは拡張された状態を扱うためのものであり、
	 * 通常はcompleteのみが返される。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param filter 検索用フィルター
	 * @return 作業項目状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItemStates(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public WorkItemState[] getWorkItemStates(String session, String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getWorkItemStates,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "filter = [" + filter + "])";
	
		try {
			checkWorkItemID(wid);
			
			DataAccessManager.begin(false);
				
			validateSession(session);
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			WorkItemState[] states = container.getWorkItemStates(wid, filter);
				
			DataAccessManager.commit();
	
			return states;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目状態の取得に失敗しました。
			String E0176 = StringManager.get("E0176") + argInfo;
			eLog.error(E0176, ex);
			throw new InvalidWorkItemException(E0176, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}
	
	/**
	 * 作業項目状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param state 作業項目状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#changeWorkItemState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.WorkItemState)
	 */
	public void changeWorkItemState(String session, String wid, WorkItemState state)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidStateException, TransitionNotAllowedException
	{
		String argInfo = "(NAUTICA API = changeWorkItemState,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "state = [" + state + "])";
	
		try {
			checkWorkItemID(wid);
			
			if (state == null) {
				// 作業項目状態が未設定です。
				String E0191 = StringManager.get("E0191");
				throw new InvalidStateException(E0191);
			}
		
			DataAccessManager.begin(false);
		
			validateSession(session);
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			String aid = container.getWorkItem(wid).getActivityID();
			container.changeWorkItemState(session, wid, state);
				
			DataAccessManager.commit();

			// 監査ログ出力
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMChangedWorkitemState", session),
		            new AuditSuffix());
		    data.setDataElement("ActivityInstanceID", aid);
		    data.setDataElement("WorkItemID", wid);
		    data.setDataElement("WorkItemState", state);
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidStateException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (TransitionNotAllowedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			// 作業項目状態の変更に失敗しました。
			String E0177 = StringManager.get("E0177") + argInfo;
			eLog.error(E0177, ex);
			throw new InvalidWorkItemException(E0177, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}
	
	/**
	 * 作業項目の再割り当てを行う。
	 *
	 * @param session セッションID
	 * @param sourceUser 割り当て前のユーザ
	 * @param targetUser 割り当て後のユーザ
	 * @param wid 作業項目ID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidSourceUserException 割り当て前ユーザが有効でない場合
	 * @throws InvalidTargetUserException 割り当て先ユーザが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#reassignWorkItem(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Participant, jp.co.argo21.nautica.workflow.wfmc.Participant, java.lang.String)
	 */
	public void reassignWorkItem(String session,
			Participant sourceUser, Participant targetUser, String wid)
	throws InvalidSessionException, InvalidWorkItemException,
	InvalidSourceUserException, InvalidTargetUserException
	{
		String argInfo = "(NAUTICA API = reassignWorkItem,"
			 + "session = [" + session + "],"
			 + "sourceUser = [" + sourceUser + "],"
			 + "targetUser = [" + targetUser + "],"
			 + "work item id = [" + wid + "])";
	
		try {
			if (sourceUser == null) {
				// 割り当て前ユーザが未設定です。
				String E0195 = StringManager.get("E0195");
				throw new InvalidSourceUserException(E0195);
			}

			if (targetUser == null) {
				// 割り当て後ユーザが未設定です。
				String E0196 = StringManager.get("E0196");
				throw new InvalidTargetUserException(E0196);
			}

			checkWorkItemID(wid);

			DataAccessManager.begin(false);
				
			validateSession(session);

			// 監査ログ出力
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMReassignedWorkitem", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);

			data.setDataElement("ActivityInstanceID", container.getWorkItem(wid).getActivityID());

			container.reassignWorkItem(wid, sourceUser, targetUser);
				
			DataAccessManager.commit();

		    data.setDataElement("WorkItemID", wid);
		    data.setDataElement("TargetDomainID", "nautica");
		    data.setDataElement("TargetNodeID", SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_SCOPE));
		    data.setDataElement("SourceUserID", sourceUser.getID());
		    data.setDataElement("TargetUserID", targetUser.getID());
		    audit.info(data);

		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidTargetUserException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目の割り当てに失敗しました。
			String E0178 = StringManager.get("E0178") + argInfo;
			eLog.error(E0178, ex);
			throw new InvalidWorkItemException(E0178, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}
	
	/**
	 * 作業項目属性一覧の取得を行う。
	 * 利用可能なフィルターはAttributeFilterのみとする。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param filter 検索用フィルター
	 * @return 作業項目属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItemAttributes(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public Attribute[] getWorkItemAttributes(String session, String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException
	{
		String argInfo = "(NAUTICA API = getWorkItemAttributes,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "filter = [" + filter + "])";
	
		try {
			checkWorkItemID(wid);

			DataAccessManager.begin(false);
				
			validateSession(session);
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			Attribute[] attrs = container.getWorkItemAttributes(session, wid, filter);
				
			DataAccessManager.commit();
	
			return attrs;
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidFilterException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目属性の取得に失敗しました。
			String E0179 = StringManager.get("E0179") + argInfo;
			eLog.error(E0179, ex);
			throw new InvalidWorkItemException(E0179, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}
	
	/**
	 * 作業項目属性の取得を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#getWorkItemAttributeValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute getWorkItemAttributeValue(String session, String wid, String name)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException
	{
		String argInfo = "(NAUTICA API = getWorkItemAttributeValue,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "attr name = [" + name + "])";
	
		try {
			checkWorkItemID(wid);
			
			if (name == null || name.trim().equals("")) {
				// 作業項目属性名が未設定です。
				String E0194 = StringManager.get("E0194");
				throw new InvalidAttributeException(E0194);
			}
			
			DataAccessManager.begin(false);
				
			validateSession(session);
			
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
				
			DataAccessManager.commit();
	
			return container.getWorkItemAttributeValue(wid, name);
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目属性の取得に失敗しました。
			String E0179 = StringManager.get("E0179") + argInfo;
			eLog.error(E0179, ex);
			throw new InvalidAttributeException(E0179, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}

	/**
	 * 作業項目属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler#assignWorkItemAttribute(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	public void assignWorkItemAttribute(String session, String wid, Attribute attr)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException, AttributeAssignmentFailedException
	{
		String argInfo = "(NAUTICA API = assignWorkItemAttribute,"
			 + "session = [" + session + "],"
			 + "work item id = [" + wid + "],"
			 + "attr = [" + attr + "])";
	
		try {
			checkWorkItemID(wid);
			
			if (attr == null) {
				// 作業項目属性が未設定です。
				String E0193 = StringManager.get("E0193");
				throw new InvalidAttributeException(E0193);
			}
			
			DataAccessManager.begin(false);
				
			validateSession(session);

			// 監査ログ出力
		    AuditDataRecord data = new AuditDataRecord(
		            new AuditPrefix("", "WMAssignedWorkitemAttributes", session),
		            new AuditSuffix());
				
			WorkItemPool pool = WorkItemPool.getInstance();
			WorkItemContainer container = pool.getWorkItemContainer(session);
			container.assignWorkItemAttribute(session, wid, attr);

			data.setDataElement("ActivityInstanceID", container.getWorkItem(wid).getActivityID());
				
			DataAccessManager.commit();

			data.setDataElement("WorkItemID", wid);
		    data.setDataElement("AttributeName", attr.getName());
		    data.setDataElement("AttributeType", attr.getType());
		    data.setDataElement("NewAttributeValue", attr.getValue());
		    audit.info(data);
	
		} catch (InvalidSessionException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidWorkItemException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (InvalidAttributeException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (AttributeAssignmentFailedException ex) {
			rollbackAndLogging(ex, argInfo);
			throw ex;
		} catch (Exception ex) {
			try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
			// 作業項目属性の割り当てに失敗しました。
			String E0180 = StringManager.get("E0180") + argInfo;
			eLog.error(E0180, ex);
			throw new InvalidWorkItemException(E0180, ex);
		} finally {
			try { DataAccessManager.close(); } catch (Exception ex) { /* Ignore */ }
		}
	}
	
	/**
	 * ロールバック処理とエラーログの出力を行う。
	 *
	 * @param ex 例外
	 * @param argInfo 引数情報
	 */
	private void rollbackAndLogging(Exception ex, String argInfo)
	{
		try { DataAccessManager.rollback(); } catch (Exception ex2) { /* Ignore */ }
		String msg = ex.getMessage() + argInfo;
		eLog.error(msg, ex);
	}
	
	/**
	 * 作業項目IDを検証する。
	 *
	 * @param wid 作業項目ID
	 * @throws InvalidWorkItemException 無効な作業項目IDの場合
	 */
	private void checkWorkItemID(String wid)
	throws InvalidWorkItemException
	{
		if (wid == null || wid.trim().equals("")) {
			// 作業項目IDが未設定です。
			String E0192 = StringManager.get("E0192");
			throw new InvalidWorkItemException(E0192);
		}
	}

	/**
	 * セッションを検証する。
	 *
	 * @param session セッションID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 */
	private void validateSession(String session)
	throws InvalidSessionException
	{
		if (session == null || session.trim().equals("")) {
			// セッションIDが未設定です。
			String E0170 = StringManager.get("E0170");
			throw new InvalidSessionException(E0170);
		}

		SessionManagerFactory factory = SessionManagerFactory.getInstance();
		SessionManager manager = factory.getSessionManager();
		manager.validateSession(session);
	}
}
