/* $Id: ResourceManagerFactory.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.omg.WorkflowException;
import jp.co.argo21.nautica.workflow.security.OrganizationManager;
import jp.co.argo21.nautica.workflow.security.RoleManager;
import jp.co.argo21.nautica.workflow.security.UserManager;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowSecurityException;

/**
 * 本ワークフローシステム運用基盤から使われるインタフェースである。
 * ResourceManagerFactoryは、各種管理をとりだすためのファクトリークラスである。
 * 任意のリソース管理システムを利用するために、このクラスを利用して具体実装を
 * 取得しなければならない。
 * 対応するユーザ管理、組織管理、権限管理もそれぞれ個別実装を持つものとする。
 * これらは順にNautica.userManager、Nautica.organizationManager、
 * Nautica.roleManagerの各プロパティで指定されたクラスを使用する。
 * 本ワークフローシステム運用基盤では、参考実装を提供する。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class ResourceManagerFactory
{
	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	/** セキュリティログ */
	private static Logger sLog = LogManager.getSercurityLogger();
	
	/** ResourceManagerFactoryのシングルトン */
	private static ResourceManagerFactory singleton;

	/** ユーザ管理 */
	private UserManager userManager;

	/** 組織管理 */
	private OrganizationManager orgManager;

	/** 権限管理 */
	private RoleManager roleManager;
	
	/**
	 * ResourceManagerFactoryの生成と初期化を行う。
	 *
	 * @throws Exception 任意の例外
	 */
	static void create() throws Exception {

		singleton = new ResourceManagerFactory();
		try {
			singleton.init();
		} catch (Exception ex) {
			eLog.error(StringManager.get("E1005"));
			throw ex;
		}
	}

	/**
	 * ResourceManagerFactoryのインスタンスを返す。
	 *
	 * @return ResourceManagerFactory
	 */
	static ResourceManagerFactory getInstance() {
		return singleton;
	}
	
	/**
	 * ResourceManagerFactoryを生成する。
	 */
	private ResourceManagerFactory()
	{
	}
	
	/**
	 * ResourceManagerFactoryを初期化する。
	 *
	 * @throws Exception 任意の例外
	 */
	private void init() throws Exception
	{
		String umcn = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_USER_MANAGER);
		if (umcn == null || umcn.trim().equals("")) {
			String E1002 = StringManager.get("E1002");
			sLog.error(E1002);
			throw new WorkflowException(E1002);
		}
		userManager = (UserManager)Class.forName(umcn).newInstance();
		String I1002 = StringManager.get("I1002");
		eLog.info(I1002);

		String omcn = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_ORG_MANAGER);
		if (omcn == null || omcn.trim().equals("")) {
			String E1003 = StringManager.get("E1003");
			sLog.error(E1003);
			throw new WorkflowException(E1003);
		}
		orgManager = (OrganizationManager)Class.forName(omcn).newInstance();
		String I1003 = StringManager.get("I1003");
		eLog.info(I1003);

		String rmcn = SystemChecker.getProperty(WorkflowEngineConstants.NAUTICA_ROLE_MANAGER);
		if (rmcn == null || rmcn.trim().equals("")) {
			String E1004 = StringManager.get("E1004");
			sLog.error(E1004);
			throw new WorkflowException(E1004);
		}
		roleManager = (RoleManager)Class.forName(rmcn).newInstance();
		String I1004 = StringManager.get("I1004");
		eLog.info(I1004);
	}
	
	/**
	 * ユーザ管理を返す。
	 *
	 * @return ユーザ管理
	 * @throws WorkflowSecurityException セキュリティ例外
	 */
	public UserManager getUserManager()
	throws WorkflowSecurityException
	{
		return userManager;
	}

	/**
	 * 組織管理を返す。
	 *
	 * @return 組織管理
	 * @throws WorkflowSecurityException セキュリティ例外
	 */
	public OrganizationManager getOrganizationManager()
	throws WorkflowSecurityException
	{
		return orgManager;
	}

	/**
	 * 権限管理を返す。
	 *
	 * @return 権限管理
	 * @throws WorkflowSecurityException セキュリティ例外
	 */
	public RoleManager getRoleManager()
	throws WorkflowSecurityException
	{
		return roleManager;
	}
}
