/* $Id: ProcessManagerFactory.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.engine;

import java.util.Hashtable;
import java.util.Map;

import jp.co.argo21.nautica.workflow.dataaccess.ActivityBean;
import jp.co.argo21.nautica.workflow.dataaccess.ActivityDAO;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessBean;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessDAO;
import jp.co.argo21.nautica.workflow.dataaccess.WorkflowDAOFactory;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Activity;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessDefinitionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidProcessInstanceException;
import jp.co.argo21.nautica.workflow.wfmc.Process;

import org.apache.log4j.Logger;

/**
 * プロセスマネージャのファクトリである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
class ProcessManagerFactory
{
	/** エンジンログ */
	private static Logger eLog = LogManager.getEngineLogger();
	
	/** ProcessManagerFactoryのシングルトン */
	private static ProcessManagerFactory singleton;

	/** プロセス管理用ハッシュテーブル */
	private Map<String, ProcessManager> processManagers = new Hashtable<String, ProcessManager>();
	
	/**
	 * ProcessManagerFactoryの生成と初期化を行う。
	 *
	 * @throws Exception 任意の例外
	 */
	static void create() throws Exception
	{
		try {
			singleton = new ProcessManagerFactory();
			singleton.init();
		} catch (Exception ex) {
			// プロセスマネージャの初期化に失敗しました。
			String E0112 = StringManager.get("E0112");
			eLog.error(E0112);
			throw ex;
		}
	}

	/**
	 * ProcessManagerFactoryのインスタンスを返す。
	 *
	 * @return ProcessManagerFactory
	 */
	static ProcessManagerFactory getInstance()
	{
		return singleton;
	}
	
	/**
	 * ProcessManagerFactoryを生成する。
	 */
	private ProcessManagerFactory()
	{
	}
	
	/**
	 * ProcessManagerFactoryを初期化する。
	 *
	 * @throws Exception 任意の例外
	 */
	private void init() throws Exception
	{
	}

	/**
	 * プロセスマネージャを返す。
	 *
	 * @return プロセスマネージャ
	 */
	public synchronized ProcessManager getProcessManager(String pdid)
	throws InvalidProcessDefinitionException
	{
		ProcessManager pm = (ProcessManager)processManagers.get(pdid);
		if (pm == null) {
			synchronized (processManagers) {
				pm = new ProcessManager(pdid);
				processManagers.put(pdid, pm);
			}
		}
		return pm;
	}

	/**
	 * 指定されたプロセスを管理するプロセスマネージャを返す。
	 *
	 * @param pid
	 * @return プロセスマネージャ
	 * @throws InvalidProcessInstanceException
	 */
	public synchronized ProcessManager getOwnerProcessManager(String pid)
	throws InvalidProcessInstanceException
	{
		try {
			ProcessManager pm = null;
			for (String key : processManagers.keySet()) {
				pm = (ProcessManager)processManagers.get(key);
				if (pm.isMemberOfProcess(pid)) {
					return pm;
				}
			}
			
			pm = createOwnerProcessManager(pid);
			
			return pm;
		} catch (InvalidProcessInstanceException ex) {
			throw ex;
		} catch (Exception ex) {
			// プロセスマネージャの取得に失敗しました。
			String E0163 = StringManager.get("E0163") + "(ProcessID = "+ pid + ")";
			throw new InvalidProcessInstanceException(E0163, ex);
		}
	}

	/**
	 * フィルターで特定されるプロセスを返す。
	 *
	 * @param filter フィルター
	 * @return プロセス
	 * @throws InvalidFilterException 無効なフィルターの場合
	 */
	Process[] getProcesses(Filter filter)
	throws InvalidFilterException
	{
		try {
			WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
			ProcessDAO procDAO = daoFactory.getProcessDAO();

			ProcessBean[] beans = procDAO.findByFilter(filter);
			return beans;
		} catch (InvalidFilterException ex) {
			throw ex;
		} catch (Exception ex) {
			// プロセスの取得に失敗しました。
			String E0151 = StringManager.get("E0151");
			throw new InvalidFilterException(E0151, ex);
		}
	}

	/**
	 * フィルターで特定されるアクティビティを返す。
	 *
	 * @param filter フィルター
	 * @return アクティビティ
	 * @throws InvalidFilterException 無効なフィルターの場合
	 */
	Activity[] getActivities(Filter filter)
	throws InvalidFilterException
	{
		try {
			WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
			ActivityDAO actDAO = daoFactory.getActivityDAO();

			ActivityBean[] beans = actDAO.findByFilter(filter);
			return beans;
		} catch (InvalidFilterException ex) {
			throw ex;
		} catch (Exception ex) {
			// アクティビティの取得に失敗しました。
			String E0124 = StringManager.get("E0124");
			throw new InvalidFilterException(E0124, ex);
		}
	}

	/**
	 * プロセスIDから所有者となるプロセスマネージャを帰す。
	 *
	 * @param pid プロセスID
	 * @return プロセスマネージャ
	 * @throws InvalidProcessInstanceException プロセスが存在しない場合
	 */
	private ProcessManager createOwnerProcessManager(String pid)
	throws InvalidProcessInstanceException
	{
		try {
			WorkflowDAOFactory daoFactory = DataAccessManager.getDAOFactory();
			ProcessDAO procDAO = daoFactory.getProcessDAO();

			ProcessBean bean = procDAO.findByPrimaryKey(pid);
			
			if (bean == null) {
				// プロセスが存在しません。
				String E0164 = StringManager.get("E0164") + "(ProcessID = " + pid + ")";
				throw new InvalidProcessInstanceException(E0164);
			}

			ProcessManager pm = new ProcessManager(bean.getProcDefinitionID());
			
			if (pm == null) {
				// プロセスマネージャを生成できませんでした。
				String E0165 = StringManager.get("E0165") + "(ProcessID = " + pid + ")";
				throw new InvalidProcessInstanceException(E0165);
			}
			synchronized (processManagers) {
				processManagers.put(bean.getProcDefinitionID(), pm);
			}
			return pm;
		} catch (InvalidProcessInstanceException ex) {
			throw ex;
		} catch (Exception ex) {
			// プロセスマネージャの生成に失敗しました。
			String E0149 = StringManager.get("E0149") + "(ProcessID = " + pid + ")";
			throw new InvalidProcessInstanceException(E0149, ex);
		}
	}
}
