/* $Id: WorkItemHandlerBean.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ejb;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.AttributeAssignmentFailedException;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidAttributeException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSessionException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidSourceUserException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidStateException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidTargetUserException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.Participant;
import jp.co.argo21.nautica.workflow.wfmc.TransitionNotAllowedException;
import jp.co.argo21.nautica.workflow.wfmc.WorkItem;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler;
import jp.co.argo21.nautica.workflow.wfmc.WorkItemState;

/**
 * WorkItemHandlerBeanは、ローカルのWorkItemHandlerに対する
 * リモート呼び出しを実現する具体実装クラスである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 * @see jp.co.argo21.nautica.workflow.wfmc.WorkItemHandler
 */
@Stateless
@Remote(WorkItemHandler.class)
@TransactionManagement(TransactionManagementType.CONTAINER)
public class WorkItemHandlerBean
implements WorkItemHandler
{
	/**
	 * 作業項目一覧の取得を行う。
	 *
	 * @param session セッションID
	 * @param filter 検索用フィルター
	 * @return 作業項目一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#getWorkItems(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public WorkItem[] getWorkItems(String session, Filter filter)
	throws InvalidSessionException, InvalidFilterException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		return itemHandler.getWorkItems(session, filter);
	}

	/**
	 * 作業項目の取得を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @return 作業項目
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#getWorkItem(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public WorkItem getWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		return itemHandler.getWorkItem(session, wid);
	}

	/**
	 * 作業項目の完了を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#completeWorkItem(java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void completeWorkItem(String session, String wid)
	throws InvalidSessionException, InvalidWorkItemException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		itemHandler.completeWorkItem(session, wid);
	}

	/**
	 * 現在の作業項目が遷移可能な状態の一覧を返す。
	 * 状態がない場合は、空の配列を返す。
	 * 利用可能なフィルターはStateFilterのみとする。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param filter 検索用フィルター
	 * @return 作業項目状態一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#getWorkItemStates(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public WorkItemState[] getWorkItemStates(String session,
			String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		return itemHandler.getWorkItemStates(session, wid, filter);
	}
	
	/**
	 * 作業項目状態の変更を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param state 作業項目状態
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidStateException 指定された状態が有効でない場合
	 * @throws TransitionNotAllowedException 状態の変更が許可されていない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#changeWorkItemState(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.WorkItemState)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void changeWorkItemState(String session,
			String wid, WorkItemState state)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidStateException, TransitionNotAllowedException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		itemHandler.changeWorkItemState(session, wid, state);
	}
	
	/**
	 * 作業項目の再割り当てを行う。
	 *
	 * @param session セッションID
	 * @param sourceUser 割り当て前のユーザ
	 * @param targetUser 割り当て後のユーザ
	 * @param wid 作業項目ID
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidSourceUserException 割り当て前ユーザが有効でない場合
	 * @throws InvalidTargetUserException 割り当て先ユーザが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#reassignWorkItem(java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Participant, jp.co.argo21.nautica.workflow.wfmc.Participant, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void reassignWorkItem(String session,
			Participant sourceUser, Participant targetUser, String wid)
	throws InvalidSessionException, InvalidWorkItemException,
	InvalidSourceUserException, InvalidTargetUserException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		itemHandler.reassignWorkItem(session, sourceUser, targetUser, wid);
	}
	
	/**
	 * 作業項目属性一覧の取得を行う。
	 * 利用可能なフィルターはNameFilter, AttributeFilter, GroupFilterとする。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param filter 検索用フィルター
	 * @return 作業項目属性一覧
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#getWorkItemAttributes(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Attribute[] getWorkItemAttributes(String session,
			String wid, Filter filter)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidFilterException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		return itemHandler.getWorkItemAttributes(session, wid, filter);
	}
	
	/**
	 * 作業項目属性の取得を行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param name 属性名
	 * @return 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#getWorkItemAttributeValue(java.lang.String, java.lang.String, java.lang.String)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Attribute getWorkItemAttributeValue(String session,
			String wid, String name)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		return itemHandler.getWorkItemAttributeValue(session, wid, name);
	}

	/**
	 * 作業項目属性の割り当てを行う。
	 *
	 * @param session セッションID
	 * @param wid 作業項目ID
	 * @param attr 属性
	 * @throws InvalidSessionException 指定されたセッションが無効の場合
	 * @throws InvalidWorkItemException 作業項目IDが有効でない場合
	 * @throws InvalidAttributeException 属性が有効でない場合
	 * @throws AttributeAssignmentFailedException 属性の割り当てに失敗した場合
	 * @ 通信障害などの場合
	 * @see jp.co.argo21.nautica.workflow.rmi.WorkItemHandlerRemote#assignWorkItemAttribute(java.lang.String, java.lang.String, jp.co.argo21.nautica.workflow.wfmc.Attribute)
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public void assignWorkItemAttribute(String session,
			String wid, Attribute attr)
	throws InvalidSessionException, InvalidWorkItemException,
		InvalidAttributeException, AttributeAssignmentFailedException
	{
		WorkItemHandler itemHandler = ComponentUtils.getWorkItemHandler(session);
		itemHandler.assignWorkItemAttribute(session, wid, attr);
	}
}
