/* $Id: OrganizationManagerBean.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.ejb;

import javax.ejb.Remote;
import javax.ejb.Stateless;
import javax.ejb.TransactionAttribute;
import javax.ejb.TransactionAttributeType;
import javax.ejb.TransactionManagement;
import javax.ejb.TransactionManagementType;

import jp.co.argo21.nautica.workflow.security.InvalidOrganizationException;
import jp.co.argo21.nautica.workflow.security.InvalidUserException;
import jp.co.argo21.nautica.workflow.security.Organization;
import jp.co.argo21.nautica.workflow.security.OrganizationManager;
import jp.co.argo21.nautica.workflow.security.User;
import jp.co.argo21.nautica.workflow.wfmc.WorkflowSecurityException;

/**
 * OrganizationManagerBean は、ローカルの OrganizationManager に対する
 * リモート呼び出しを実現する具体実装クラスである。
 *
 * @author  ysahara(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
@Stateless
@Remote(OrganizationManager.class)
@TransactionManagement(TransactionManagementType.CONTAINER)
public class OrganizationManagerBean implements OrganizationManager
{
	/**
	 * oidに対応する組織を返す。
	 * ユーザが存在しない場合は空の配列を返す。
	 *
	 * @param oid 組織ID
	 * @return 組織単位
	 * @throws InvalidOrganizationException 指定された組織が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Organization getOrganization(String oid)
	throws InvalidOrganizationException, WorkflowSecurityException
	{
		OrganizationManager om = ComponentUtils.getOrganizationManager();
		return om.getOrganization(oid);
	}

	/**
	 * 該当組織に属しているユーザの一覧を返す。
	 * ユーザが存在しない場合は空の配列を返す。
	 *
	 * @param org 組織単位
	 * @return ユーザ一覧
	 * @throws InvalidOrganizationException 指定された組織が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public User[] getAssignedUser(Organization org)
	throws InvalidOrganizationException, WorkflowSecurityException
	{
		OrganizationManager om = ComponentUtils.getOrganizationManager();
		return om.getAssignedUser(org);
	}

	/**
	 * 該当ユーザの所属する組織を返す。
	 * 組織が存在しない場合は空の配列を返す。
	 * 
	 * @param user ユーザ
	 * @return 組織一覧
	 * @throws InvalidUserException 指定されたユーザが有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Organization[] getOrganizationByUser(User user)
	throws InvalidUserException, WorkflowSecurityException
	{
		OrganizationManager om = ComponentUtils.getOrganizationManager();
		return om.getOrganizationByUser(user);
	}
	
	/**
	 * 該当組織の上位組織を返す。
	 * 上位組織が存在しない場合は null を返す。
	 *
	 * @param org 組織単位
	 * @return 上位組織
	 * @throws InvalidOrganizationException 指定された組織が有効でない場合
	 * @throws WorkflowSecurityException その他のセキュリティ例外
	 */
	@TransactionAttribute(TransactionAttributeType.REQUIRED)
	public Organization getHigherOrganization(Organization org)
	throws InvalidOrganizationException, WorkflowSecurityException
	{
		OrganizationManager om = ComponentUtils.getOrganizationManager();
		return om.getHigherOrganization(org);
	}
}
