/* $Id: ActivityDefinitionImpl.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 * 
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.definition.impl;

import java.util.ArrayList;
import java.util.List;

import org.wfmc.x2002.xpdl10.ActivityDocument;
import org.wfmc.x2002.xpdl10.ActualParametersDocument;
import org.wfmc.x2002.xpdl10.BlockActivityDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributeDocument;
import org.wfmc.x2002.xpdl10.ExtendedAttributesDocument;
import org.wfmc.x2002.xpdl10.ImplementationDocument;
import org.wfmc.x2002.xpdl10.SubFlowDocument;
import org.wfmc.x2002.xpdl10.ToolDocument;

import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.util.IDGenerator;
import jp.co.argo21.nautica.workflow.util.StringUtils;

/**
 * ActivityDefinitionインタフェースの実装クラスである。
 * 
 * @author  Norihiro Itoh(ARGO 21 Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ActivityDefinitionImpl implements ActivityDefinition
{
	/** シリアルバージョンUID */
	private static final long serialVersionUID = 20050000000000001L;

	/** 未初期化時の仮ID */
	private static final String UNINITIALIZED_ID = "--UNINITIALIZED--"; 

	/** ID */
	private String id;
	/** 名称 */
	private String name;
	/** 備考 */
	private String description;
	/** 実行期限 */
	private int limit;
	/** 論理X座標 */
	private int logicalX;
	/** 論理Y座標 */
	private int logicalY;
	/** ブロックID */
	private String blockID;
	/** 所属ブロックID */
	private String affiliateBID;
	/** 親ブロックID */
	private String parentBID;
	/** アプリケーションID */
	private String applicationID;
	/** サブプロセスID */
	private String subprocessID;
	/** 実行許可者ID */
	private String performerID;
	/** サブプロセス実行時の同期状態 */
	private int sync;
	/** アプリケーションパラメータ一覧 */
	private List<String> applicationParameters = new ArrayList<String>();
	/** サブプロセスパラメータ一覧 */
	private List<String> subprocessParameters = new ArrayList<String>();
	/** アクティビティ型 */
	private String type;
	/** エンジン名 */
	private String engineName;

	/**
	 * アクティビティを生成する。
	 * 
	 * @param act	XmlObject
	 */
	public ActivityDefinitionImpl(ActivityDocument.Activity act)
	{
		String id = act.getId();
		if (StringUtils.checkNull(id) || id.equals(UNINITIALIZED_ID)) {
			setID(IDGenerator.generate(ID_PREFIX));
		} else {
			setID(id);
		}
		setName(act.getName());
		setDescription(act.getDescription());

		try {
			setLimit(Integer.parseInt(act.getLimit()));
		} catch (Exception ex) {
			setLimit(0);
		}
	
		if (act.isSetBlockActivity()) {
			BlockActivityDocument.BlockActivity ba = act.getBlockActivity();
			setBlockID(ba.getBlockId());
		}
		
		if (act.isSetPerformer()) {
			setPerformerID(act.getPerformer());
		}
		
		if (act.isSetImplementation()) {
			ImplementationDocument.Implementation imp = act.getImplementation();
			if (imp.isSetSubFlow()) {
				SubFlowDocument.SubFlow sf = imp.getSubFlow();
				setSubprocessID(sf.getId());
				setExecution(sf.getExecution().intValue());
				if (sf.isSetActualParameters()) {
					ActualParametersDocument.ActualParameters aps = sf.getActualParameters();
					int count = aps.sizeOfActualParameterArray();
					for (int i = 0; i < count; i++) {
						String pid = aps.getActualParameterArray(i);
						addSubprocessParameterID(pid);
					}
				}
			}

			if (imp.sizeOfToolArray() == 1) {
				ToolDocument.Tool tool = imp.getToolArray(0);
				setApplicationID(tool.getId());
				if (tool.isSetActualParameters()) {
					ActualParametersDocument.ActualParameters aps = tool.getActualParameters();
					int count = aps.sizeOfActualParameterArray();
					for (int i = 0; i < count; i++) {
						String aid = aps.getActualParameterArray(i);
						addApplicationParameterID(aid);
					}
				}
			}
		}

		if (act.isSetExtendedAttributes()) {
			ExtendedAttributesDocument.ExtendedAttributes eas = act.getExtendedAttributes();
			int count = eas.sizeOfExtendedAttributeArray();
			for (int i = 0; i < count; i++) {
				ExtendedAttributeDocument.ExtendedAttribute ea = eas.getExtendedAttributeArray(i);
				String n = ea.getName();
				String v = ea.getValue();
				if (n.equals(LOGICAL_X)) {
					setLogicalX(Integer.parseInt(v));
				} else if (n.equals(LOGICAL_Y)) {
					setLogicalY(Integer.parseInt(v));
				} else if (n.equals(AFFILIATE_BID)) {
					setAffiliateBID(v);
				} else if (n.equals(PARENT_BID)) {
					setParentBID(v);
				} else if (n.equals(ACTIVITY_TYPE)) {
					setType(v);
				} else if (n.equals(ENGINE_NANE)) {
					setEngineName(v);
				}
			}
		}
	}

	/**
	 * ブロックIDを返す。
	 * 
	 * @return ブロックID
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getBlockID()
	 */
	public String getBlockID()
	{
		return blockID;
	}

	/**
	 * 実行期限を返す。
	 * 
	 * @return	実行期限
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getLimit()
	 */
	public int getLimit()
	{
		return limit;
	}

	/**
	 * 論理X座標を返す。
	 * 
	 * @return 論理X座標
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getLogicalX()
	 */
	public int getLogicalX()
	{
		return logicalX;
	}

	/**
	 * 論理Y座標を返す。
	 * 
	 * @return 論理Y座標
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getLogicalY()
	 */
	public int getLogicalY()
	{
		return logicalY;
	}

	/**
	 * アクティビティ型を返す。
	 * 
	 * @return	アクティビティ型
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getType()
	 */
	public String getType()
	{
		return type;
	}

	/**
	 * 説明を返す。
	 * 
	 * @return 説明
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * IDを返す。
	 * 
	 * @return ID
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getID()
	 */
	public String getID()
	{
		return id;
	}

	/**
	 * アクティビティ名称を返す。
	 * 
	 * @return アクティビティ名称
	 * @see jp.co.argo21.nautica.workflow.definition.Definition#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * アプリケーションIDを返す。
	 * 
	 * @return	アプリケーションID
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getApplicationID()
	 */
	public String getApplicationID()
	{
		return applicationID;
	}

	/**
	 * アプリケーションパラメータ一覧を返す。
	 * 
	 * @return	アプリケーションパラメータ一覧
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getApplicationParameters()
	 */
	public List<String> getApplicationParameters()
	{
		return applicationParameters;
	}

	/**
	 * サブプロセスIDを返す。
	 * 
	 * @return	サブプロセスID
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getSubprocessID()
	 */
	public String getSubprocessID()
	{
		return subprocessID;
	}

	/**
	 * サブプロセスパラメータ一覧を返す。
	 * 
	 * @return	サブプロセスパラメータ一覧
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getSubprocessParameters()
	 */
	public List<String> getSubprocessParameters()
	{
		return subprocessParameters;
	}

	/**
	 * サブプロセス実行時の同期状態を返す。
	 * 
	 * @return 同期状態
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getExecution()
	 */
	public int getExecution()
	{
		return sync;
	}

	/**
	 * 所属ブロックIDを返す。
	 * 
	 * @return	所属ブロックID
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getAffiliateBID()
	 */
	public String getAffiliateBID()
	{
		return affiliateBID;
	}

	/**
	 * 親ブロックIDを返す。
	 * 
	 * @return	親ブロックID
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getParentBID()
	 */
	public String getParentBID()
	{
		return parentBID;
	}

	/**
	 * 実行許可者IDを返す。
	 * 
	 * @return	実行許可者ID
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getPerformerID()
	 */
	public String getPerformerID()
	{
		return performerID;
	}

	/**
	 * エンジン名を返す。
	 *
	 * @return エンジン名
	 * @see jp.co.argo21.nautica.workflow.definition.ActivityDefinition#getEngineName()
	 */
	public String getEngineName()
	{
		return engineName;
	}

	/**
	 * IDを設定する。
	 * 
	 * @param id	ID
	 */	
	private void setID(String id)
	{
		this.id = id;
	}

	/**
	 * ブロックIDを設定する。
	 * 
	 * @param id ブロックID
	 */
	private void setBlockID(String id)
	{
		this.blockID = id;
	}

	/**
	 * 実行期限を設定する。単位は所属するプロセスのDurationUnitを使用する。
	 * 
	 * @param limit 待機時間の上限値
	 */
	private void setLimit(int limit)
	{
		this.limit = limit;
	}

	/**
	 * 論理X座標を設定する。
	 * 
	 * @param x 論理X座標
	 */
	private void setLogicalX(int x)
	{
		this.logicalX = x;
	}

	/**
	 * 論理Y座標を設定する。
	 * 
	 * @param y 論理Y座標
	 */
	private void setLogicalY(int y)
	{
		this.logicalY = y;
	}

	/**
	 * アクティビティの種類を設定する。
	 * 
	 * @param type	アクティビティの種類
	 */
	private void setType(String type)
	{
		this.type = type;
	}

	/**
	 * 説明を設定する。
	 * 
	 * @param desc 説明
	 */
	private void setDescription(String desc)
	{
		this.description = desc;
	}

	/**
	 * アクティビティ名称を設定する。
	 * 
	 * @param name アクティビティ名称
	 */
	private void setName(String name)
	{
		this.name = name;
	}

	/**
	 * アプリケーションIDを設定する。
	 * 
	 * @param id	アプリケーションID
	 */
	private void setApplicationID(String id)
	{
		this.applicationID = id;
	}

	/**
	 * サブプロセスIDを設定する。
	 * 
	 * @param id	サブプロセスID
	 */
	private void setSubprocessID(String id)
	{
		this.subprocessID = id;
	}

	/**
	 * サブプロセス実行時の同期状態を設定する。
	 * 
	 * @param sync 同期状態
	 */
	private void setExecution(int sync)
	{
		this.sync = sync;
	}

	/**
	 * 所属ブロックIDを設定する。
	 * 
	 * @param id	所属ブロックID
	 */
	private void setAffiliateBID(String id)
	{
		affiliateBID = id;
	}

	/**
	 * 親ブロックIDを設定する。
	 * 
	 * @param id	親ブロックID
	 */
	private void setParentBID(String id)
	{
		parentBID = id;
	}

	/**
	 * 実行許可者IDを設定する。
	 * 
	 * @param id	実行許可者ID
	 */
	private void setPerformerID(String id)
	{
		performerID = id;
	}

	/**
	 * アプリケーションパラメータを追加する。
	 * 
	 * @param id	アプリケーションパラメータ
	 */
	private void addApplicationParameterID(String id)
	{
		this.applicationParameters.add(id);
	}

	/**
	 * エンジン名を設定する。
	 * 
	 * @param engineName エンジン名
	 */
	private void setEngineName(String engineName)
	{
		this.engineName = engineName;
	}

	/**
	 * サブプロセスパラメータを追加する。
	 * 
	 * @param id	サブプロセスパラメータ
	 */
	private void addSubprocessParameterID(String id)
	{
		this.subprocessParameters.add(id);
	}
}
