/* $Id: MySQLSourceActivityDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.SourceActivityBean;
import jp.co.argo21.nautica.workflow.dataaccess.SourceActivityDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * SourceActivityBeanをデータとするMySQL用DAOクラスである。
 * <pre><tt>
 * ETAACTIVITYテーブル
 * +--------------+-------------+------+-----+---------+-------+
 * | Field        | Type        | Null | Key | Default | Extra |
 * +--------------+-------------+------+-----+---------+-------+
 * | PROCESS_ID   | varchar(64) |      | PRI |         |       |
 * | ENGINE_ID    | varchar(64) |      |     |         |       |
 * | ROOT_PROC_ID | varchar(64) |      |     |         |       |
 * | ROOT_ACT_ID  | varchar(64) |      |     |         |       |
 * +--------------+-------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLSourceActivityDAO extends AbstractDAO implements SourceActivityDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** プロセスIDのカラム名 */
	public static final String PROCESS_ID   = "PROCESS_ID";
	/** エンジンIDのカラム名 */
	public static final String ENGINE_ID    = "ENGINE_ID";
	/** ルートプロセスIDのカラム名 */
	public static final String ROOT_PROC_ID = "ROOT_PROC_ID";
	/** ルートアクティビティIDのカラム名 */
	public static final String ROOT_ACT_ID  = "ROOT_ACT_ID";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from ETAACTIVITY where PROCESS_ID = ? ";

	private static final String SQL_INSERT
	= "insert into ETAACTIVITY ("
		+ PROCESS_ID + ", "
		+ ENGINE_ID + ", "
		+ ROOT_PROC_ID + ", "
		+ ROOT_ACT_ID + ") values (?,?,?,?)";

	private static final String SQL_DELETE
	= "delete from ETAACTIVITY where PROCESS_ID = ? ";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLSourceActivityDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * 起動プロセスIDで起動元アクティビティ情報を取得する。
	 *
	 * @param pid プロセスID
	 * @return 起動元アクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.SourceActivityDAO#findByPrimaryKey(java.lang.String)
	 */
	public SourceActivityBean findByPrimaryKey(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			SourceActivityBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * 起動元アクティビティ情報を挿入する。
	 *
	 * @param bean 起動元アクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.SourceActivityDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.SourceActivityBean)
	 */
	public void insert(SourceActivityBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getProcessID());
			st.setString(2, bean.getEngineID());
			st.setString(3, bean.getRootProcID());
			st.setString(4, bean.getRootActID());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 外部プロセス起動元アクティビティ情報の挿入に失敗しました。
				String errMsg = StringManager.get("E0026");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}

	/**
	 * 起動プロセスIDで特定される起動元アクティビティ情報を削除する。
	 *
	 * @param pid プロセスID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.SourceActivityDAO#delete(java.lang.String)
	 */
	public void delete(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_DELETE);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 外部プロセス起動元アクティビティ情報の削除に失敗しました。
				String errMsg = StringManager.get("E0025");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}

	/**
	 * 結果セットから起動元アクティビティ情報を作成する。
	 *
	 * @param rs 結果セット
	 * @return 起動元アクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 */
	private SourceActivityBean createBean(ResultSet rs)
	throws SQLException
	{
		SourceActivityBean bean = new SourceActivityBean();

		bean.setProcessID(rs.getString(PROCESS_ID));
		bean.setEngineID(rs.getString(ENGINE_ID));
		bean.setRootProcID(rs.getString(ROOT_PROC_ID));
		bean.setRootActID(rs.getString(ROOT_ACT_ID));
		
		return bean;
	}
}
