/* $Id: MySQLProcessWaitActivityDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityBean;
import jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * ProcessWaitActivityBeanをデータとするMySQL用DAOクラスである。
 * <pre><tt>
 * WAITPROCACTテーブル
 * +------------+-------------+------+-----+---------+-------+
 * | Field      | Type        | Null | Key | Default | Extra |
 * +------------+-------------+------+-----+---------+-------+
 * | PROC_ID    | varchar(64) |      | MUL |         |       |
 * | ACT_ID     | varchar(64) |      |     |         |       |
 * | SUBPROC_ID | varchar(64) |      |     |         |       |
 * +------------+-------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLProcessWaitActivityDAO extends AbstractDAO implements ProcessWaitActivityDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** プロセスIDのカラム名 */
	public static final String PROC_ID    = "PROC_ID";
	/** アクティビティIDのカラム名 */
	public static final String ACT_ID     = "ACT_ID";
	/** サブプロセスIDのカラム名 */
	public static final String SUBPROC_ID = "SUBPROC_ID";

	private static final String SQL_SELECT_BY_SUBPROC_ID
	= "select * from WAITPROCACT where SUBPROC_ID = ? ";

	private static final String SQL_SELECT_BY_PROCID_AND_ACTID
	= "select * from WAITPROCACT where PROC_ID = ? AND ACT_ID = ? ";

	private static final String SQL_INSERT
	= "insert into WAITPROCACT ("
		+ PROC_ID + ", "
		+ ACT_ID + ", "
		+ SUBPROC_ID + ") values (?,?,?)";

	private static final String SQL_DELETE
	= "delete from WAITPROCACT where SUBPROC_ID = ? ";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLProcessWaitActivityDAO(Connection conn)
	{
		super(conn);
	}

	/**
	 * IDに対応するサブプロセス終了待ちアクティビティ情報を取得する。
	 *
	 * @param spid サブプロセスID
	 * @return サブプロセス終了待ちアクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityDAO#findBySubprocessID(java.lang.String)
	 */
	public ProcessWaitActivityBean findBySubprocessID(String spid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_SUBPROC_ID);
		try {
			st.setString(1, spid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ProcessWaitActivityBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * プロセスIdとアクティビティIDでサブプロセス終了待ちアクティビティ情報を取得する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @return サブプロセス終了待ちアクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityDAO#findByProcIDANDActID(java.lang.String, java.lang.String)
	 */
	public ProcessWaitActivityBean findByProcIDANDActID(String pid, String aid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PROCID_AND_ACTID);
		try {
			st.setString(1, pid);
			st.setString(2, aid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ProcessWaitActivityBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * サブプロセス終了待ちアクティビティ情報を挿入する。
	 *
	 * @param bean サブプロセス終了待ちアクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityBean)
	 */
	public void insert(ProcessWaitActivityBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getProcID());
			st.setString(2, bean.getActID());
			st.setString(3, bean.getSubprocID());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// サブプロセス終了待ちアクティビティ情報の挿入に失敗しました。
				String errMsg = StringManager.get("E0024");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * サブプロセス終了待ちアクティビティ情報を削除する。
	 *
	 * @param spid サブプロセスID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ProcessWaitActivityDAO#delete(java.lang.String)
	 */
	public void delete(String spid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_DELETE);
		try {
			st.setString(1, spid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// サブプロセス終了待ちアクティビティ情報の削除に失敗しました。
				String errMsg = StringManager.get("E0023");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットからプロセス終了待ちアクティビティ情報を作成する。
	 *
	 * @param rs 結果セット
	 * @return プロセス終了待ちアクティビティ情報
	 * @throws SQLException 任意のSQL例外
	 */
	private ProcessWaitActivityBean createBean(ResultSet rs)
	throws SQLException
	{
		ProcessWaitActivityBean bean = new ProcessWaitActivityBean();

		bean.setProcID(rs.getString(PROC_ID));
		bean.setActID(rs.getString(ACT_ID));
		bean.setSubprocID(rs.getString(SUBPROC_ID));
		
		return bean;
	}
}
