/* $Id: MySQLConcurrencyBlockDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockBean;
import jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * ConcurrencyBlockBeanをデータとするMySQL用DAOクラスである。
 * <pre><tt>
 * CONCURBLOCKテーブル
 * +----------------+-------------+------+-----+---------+-------+
 * | Field          | Type        | Null | Key | Default | Extra |
 * +----------------+-------------+------+-----+---------+-------+
 * | BLOCK_ID       | varchar(64) |      |     |         |       |
 * | UPPER_BLOCK_ID | varchar(64) | YES  |     | NULL    |       |
 * | PROC_ID        | varchar(64) |      | MUL |         |       |
 * | ACT_ID         | varchar(64) |      |     |         |       |
 * +----------------+-------------+------+-----+---------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLConcurrencyBlockDAO extends AbstractDAO implements ConcurrencyBlockDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** ブロックIDのカラム名 */
	public static final String BLOCK_ID       = "BLOCK_ID";
	/** 上位ブロックIDのカラム名 */
	public static final String UPPER_BLOCK_ID = "UPPER_BLOCK_ID";
	/** プロセスIDのカラム名 */
	public static final String PROC_ID        = "PROC_ID";
	/** アクティビティIDのカラム名 */
	public static final String ACT_ID         = "ACT_ID";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from CONCURBLOCK where PROC_ID = ? AND BLOCK_ID = ? ";

	//存在するサブブロックを探す
	private static final String SQL_SELECT_COUNT_EXIST_BY_BLOCK_ID
	= "select COUNT(*) from CONCURBLOCK where PROC_ID = ? AND UPPER_BLOCK_ID = ? ";

	//存在するサブブロックを探す
	private static final String SQL_SELECT_COUNT_EXIST_BY_PROC_ID
	= "select COUNT(*) from CONCURBLOCK where PROC_ID = ? ";

	private static final String SQL_INSERT
	= "insert into CONCURBLOCK ("
		+ BLOCK_ID + ", "
		+ UPPER_BLOCK_ID + ", "
		+ PROC_ID + ", "
		+ ACT_ID + ") values (?,?,?,?)";

	private static final String SQL_DELETE
	= "delete from CONCURBLOCK where PROC_ID = ? AND BLOCK_ID = ? ";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLConcurrencyBlockDAO(Connection conn)
	{
		super(conn);
	}

	/**
	 * 主キーにより、並行性ブロックを取得する。
	 *
	 * @param pid プロセスID
	 * @param bid ブロックID
	 * @return 並行性ブロック
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO#findByPrimaryKey(java.lang.String, java.lang.String)
	 */
	public ConcurrencyBlockBean findByPrimaryKey(String pid, String bid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, pid);
			st.setString(2, bid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			ConcurrencyBlockBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

	/**
	 * ブロック直下のサブブロックの存在を取得する。
	 *
	 * @param pid プロセスID
	 * @param bid ブロックID
	 * @return 存在するサブブロックの個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO#findSubblockExistenceByBlockID(java.lang.String, java.lang.String)
	 */
	public int findSubblockExistenceByBlockID(String pid, String bid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_COUNT_EXIST_BY_BLOCK_ID);
		try {
			st.setString(1, pid);
			st.setString(2, bid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			int count = 0;
			if (rs.next()) {
				count = rs.getInt(1);
			}
			return count;
		} finally {
			st.close();
		}
	}
	
	/**
	 * プロセス直下のサブブロックの存在を取得する。
	 *
	 * @param pid プロセスID
	 * @return 存在するサブブロックの個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO#findSubblockExistenceByProcessID(java.lang.String)
	 */
	public int findSubblockExistenceByProcessID(String pid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_COUNT_EXIST_BY_PROC_ID);
		try {
			st.setString(1, pid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			int count = 0;
			if (rs.next()) {
				count = rs.getInt(1);
			}
			return count;
		} finally {
			st.close();
		}
	}
	
	/**
	 * 並行性ブロックを挿入する。
	 *
	 * @param bean 並行性ブロック
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockBean)
	 */
	public void insert(ConcurrencyBlockBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getBlockID());
			st.setString(2, bean.getUpperBlockID());
			st.setString(3, bean.getProcID());
			st.setString(4, bean.getActID());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 並行性ブロックレコードの挿入に失敗しました。
				String errMsg = StringManager.get("E0015");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}

	/**
	 * 並行性ブロックを削除する。
	 *
	 * @param pid プロセスID
	 * @param bid ブロックID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ConcurrencyBlockDAO#delete(java.lang.String, java.lang.String)
	 */
	public void delete(String pid, String bid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_DELETE);
		try {
			st.setString(1, pid);
			st.setString(2, bid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 並行性ブロックレコードの削除に失敗しました。
				String errMsg = StringManager.get("E0014");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットから並行性ブロックを作成する。
	 *
	 * @param rs 結果セット
	 * @return 並行性ブロック
	 * @throws SQLException 任意のSQL例外
	 */
	private ConcurrencyBlockBean createBean(ResultSet rs)
	throws SQLException
	{
		ConcurrencyBlockBean bean = new ConcurrencyBlockBean();

		bean.setBlockID(rs.getString(BLOCK_ID));
		bean.setUpperBlockID(rs.getString(UPPER_BLOCK_ID));
		bean.setProcID(rs.getString(PROC_ID));
		bean.setActID(rs.getString(ACT_ID));
		
		return bean;
	}
}
