/* $Id: MySQLAppExecutionDAO.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionBean;
import jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.ta.ApplicationState;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * アプリケーション実行テーブルに対応するMySQL用DAOクラスである。。
 * 
 * <pre><tt>
 * APPEXECテーブル
 * +------------+-------------+------+-----+---------+-------+
 * | Field      | Type        | Null | Key | Default | Extra |
 * +------------+-------------+------+-----+---------+-------+
 * | APP_NAME   | varchar(64) |      | PRI |         |       |
 * | TOOL_AGENT | varchar(64) |      | PRI |         |       |
 * | EXEC_ID    | bigint(19)  |      | PRI | 0       |       |
 * | START_DATE | datetime    | YES  |     | NULL    |       |
 * | STARTER_ID | varchar(64) | YES  |     | NULL    |       |
 * | END_DATE   | datetime    | YES  |     | NULL    |       |
 * | APP_STATE  | int(1)      | YES  |     | 2       |       |
 * +------------+-------------+------+-----+---------+-------+
 * </pre>
 * 
 * @author  kiyoda(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLAppExecutionDAO extends AbstractDAO implements AppExecutionDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

    /** アプリケーション名のカラム名 */
	public static final String APP_NAME   = "APP_NAME";
	/** アプリケーションを起動したツールエージェントのクラス名のカラム名 */
	public static final String TOOL_AGENT  = "TOOL_AGENT";
	/** 実行IDのカラム名 */
	public static final String EXEC_ID    = "EXEC_ID";
	/** 開始日時のカラム名 */
	public static final String START_DATE = "START_DATE";
	/** 実行者のカラム名 */
	public static final String STARTER_ID = "STARTER_ID";
	/** 終了日時のカラム名 */
	public static final String END_DATE   = "END_DATE";
	/** 実行状態のカラム名 */
	public static final String APP_STATE  = "APP_STATE";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from APPEXEC where APP_NAME = ? and TOOL_AGENT = ? and EXEC_ID = ?";

	private static final String SQL_SELECT_BY_EXEC_ID
	= "select * from APPEXEC where EXEC_ID = ? ";

	private static final String SQL_SELECT_BY_STARTER_ID
	= "select * from APPEXEC where STARTER_ID = ? ";

	private static final String SQL_INSERT
	= "insert into APPEXEC ("
		+ APP_NAME + ", "
		+ TOOL_AGENT + ", "
		+ EXEC_ID + ", "
		+ START_DATE + ", "
		+ STARTER_ID + ", "
		+ END_DATE + ", "
		+ APP_STATE + ") values (?, ?, ?, ?, ?, ?, ?)";

	private static final String SQL_UPDATE_STATE
	= "update APPEXEC set "
		+ APP_STATE + " = ? where "
		+ APP_NAME + " = ? and "
		+ TOOL_AGENT + " = ? and "
		+ EXEC_ID + " = ? and "
		+ APP_STATE + " = ?";

	private static final String SQL_UPDATE_STATE_END_DATE
	= "update APPEXEC set "
		+ APP_STATE + " = ?, " 
		+ END_DATE + " = ? where "
		+ APP_NAME + " = ? and "
		+ TOOL_AGENT + " = ? and "
		+ EXEC_ID + " = ? and "
		+ APP_STATE + " = ?";
	
	/**
	 * アプリケーション実行テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
    MySQLAppExecutionDAO(Connection conn)
	{
		super(conn);
	}

    /**
     * 主キーで、アプリケーション実行テーブルオブジェクトを取得する。
     * 
     * アプリケーション実行テーブルの主キーである、
     * アプリケーション名、ツールエージェントのクラス名、実行IDで、
     * アプリケーション実行テーブルオブジェクトを取得する。
     *
     * @param appName		アプリケーション名
     * @param toolAgent		アプリケーションを起動したツールエージェントのクラス名
     * @param execID		ツールエージェントが発行した実行ID
     * @return		アプリケーション実行テーブルオブジェクト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
     * @see jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO#findByPrimaryKey(java.lang.String, java.lang.String, int)
     */
    public AppExecutionBean findByPrimaryKey(String appName, String toolAgent, int execID) 
    throws SQLException, DAOException 
    {
        AppExecutionBean bean = null;
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
		    st.setString(1, appName);
		    st.setString(2, toolAgent);
		    st.setInt(3, execID);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/

			ResultSet rs = st.executeQuery();
			
			if (rs.next()) {
				bean = createBean(rs);
			}
		} finally {
			st.close();
		}
		
        return bean;
    }

    /**
	 * 実行IDで、アプリケーション実行テーブルオブジェクトを取得する。
	 * 
	 * 主キーの一部である、実行IDでアプリケーション実行テーブルオブジェクトのリストを
	 * 取得する
	 *
	 * @param execID	実行ID
	 * @return		アプリケーション実行テーブルオブジェクトのリスト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
     * @see jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO#findByExecID(int)
     */
	public List<AppExecutionBean> findByExecID(int execID)
	throws SQLException, DAOException 
	{
		List<AppExecutionBean> list = new ArrayList<AppExecutionBean>();
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_EXEC_ID);
		try {
		    st.setInt(1, execID);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/

			ResultSet rs = st.executeQuery();
			
			while (rs.next()) {
				AppExecutionBean bean = createBean(rs);
				list.add(bean);
			}
		} finally {
			st.close();
		}
		return list;
	}
    
	/**
	 * 開始者IDで、アプリケーション実行テーブルオブジェクトを取得する。
	 * 開始者IDとは、対応する作業項目IDである。
	 *
	 * @param starterID 開始者ID
	 * @return		アプリケーション実行テーブルオブジェクトのリスト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO#findByStarterID(java.lang.String)
	 */
	public List<AppExecutionBean> findByStarterID(String starterID)
	throws SQLException, DAOException
	{
		List<AppExecutionBean> list = new ArrayList<AppExecutionBean>();
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_STARTER_ID);
		try {
		    st.setString(1, starterID);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/

		    ResultSet rs = st.executeQuery();
			
			while (rs.next()) {
				AppExecutionBean bean = createBean(rs);
				list.add(bean);
			}
		} finally {
			st.close();
		}
		return list;
	}
    
    /**
	 * アプリケーション実行テーブルにレコードを追加する。
	 * 
	 * @param bean		アプリケーション実行テーブルオブジェクト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
     * @see jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.AppExecutionBean)
     */
    public void insert(AppExecutionBean bean) throws SQLException, DAOException {
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getAppName());
			st.setString(2, bean.getToolAgent());
			st.setInt(3, bean.getExecId());
			st.setTimestamp(4, bean.getStartDate());
			st.setString(5, bean.getStarterId());
			st.setTimestamp(6, bean.getEndDate());
			st.setInt(7, bean.getAppState().toInt());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 挿入エラー
			    String errMsg = StringManager.get("E0005");
			    errMsg = errMsg 
			    	+ "APPNAME=" + bean.getAppName() + ", "
		        	+ "TOOL_AGENT=" + bean.getToolAgent() + ", "
		        	+ "EXEC_ID=" + bean.getExecId() + ", "
		        	+ "START_DATE=" + bean.getStartDate() + ", "
		        	+ "STARTER_ID=" + bean.getStarterId() + ", "
		        	+ "END_DATE=" + bean.getEndDate() + ", "
		        	+ "APP_STATE=" + bean.getAppState();
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
    }

    /**
	 * 実行状態を更新する
	 *
	 * @param appName		アプリケーション名
	 * @param toolAgent		ツールエージェントのクラス名
	 * @param execID		実行ID
	 * @param oldState		旧実行状態
	 * @param newState		新実行状態
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO#updateState(java.lang.String, java.lang.String, int, jp.co.argo21.nautica.workflow.ta.ApplicationState, jp.co.argo21.nautica.workflow.ta.ApplicationState)
	 */
	public void updateState(String appName, String toolAgent, int execID, 
	        ApplicationState oldState, ApplicationState newState)
            throws SQLException, DAOException {
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_STATE);
		try {
			st.setInt(1, newState.toInt());
			st.setString(2, appName);
			st.setString(3, toolAgent);
			st.setInt(4, execID);
			st.setInt(5, oldState.toInt());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 更新エラー
			    String errMsg = StringManager.get("E0006");
			    errMsg = errMsg
			    	+ "APPNAME=" + appName + ", "
			    	+ "TOOL_AGENT=" + toolAgent + ", "
			    	+ "EXEC_ID=" + execID + ", "
			    	+ "OLD_STATE=" + oldState + ", "
			    	+ "NEW_STATE=" + newState;
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
    }
	
	/**
	 * 実行状態と終了日時を更新する
	 *
	 * @param appName		アプリケーション名
	 * @param toolAgent		ツールエージェントのクラス名
	 * @param execID		実行ID
	 * @param oldState		旧実行状態
	 * @param newState		新実行状態
	 * @param newDate		新終了日時
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.AppExecutionDAO#updateState(java.lang.String, java.lang.String, int, jp.co.argo21.nautica.workflow.ta.ApplicationState, jp.co.argo21.nautica.workflow.ta.ApplicationState, java.sql.Timestamp)
	 */
	public void updateState(String appName, String toolAgent, int execID, 
	        ApplicationState oldState, ApplicationState newState, Timestamp newDate)
	throws SQLException, DAOException {
	    PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_STATE_END_DATE);
		try {
			st.setInt(1, newState.toInt());
			st.setTimestamp(2, newDate);
			st.setString(3, appName);
			st.setString(4, toolAgent);
			st.setInt(5, execID);
			st.setInt(6, oldState.toInt());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 更新エラー
			    String errMsg = StringManager.get("E0006");
			    errMsg = errMsg
			    	+ "APPNAME=" + appName + ", "
			    	+ "TOOL_AGENT=" + toolAgent + ", "
			    	+ "EXEC_ID=" + execID + ", "
			    	+ "OLD_STATE=" + oldState + ", "
			    	+ "NEW_STATE=" + newState + ", "
			    	+ "NEW_END_DATE=" + newDate;
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}

	/**
	 * ResultSet から アプリケーション実行テーブルオブジェクトを生成する
	 *
	 * @param rs	ResultSet
	 * @return		アプリケーション実行テーブルオブジェクト
	 * @throws SQLException 任意のSQL例外
	 */
	private AppExecutionBean createBean(ResultSet rs)
	throws SQLException
	{
	    AppExecutionBean bean = new AppExecutionBean();

	    bean.setAppName(rs.getString(APP_NAME));
	    bean.setToolAgent(rs.getString(TOOL_AGENT));
	    bean.setExecId(rs.getInt(EXEC_ID));
	    bean.setStartDate(rs.getTimestamp(START_DATE));
	    bean.setStarterId(rs.getString(STARTER_ID));
	    bean.setEndDate(rs.getTimestamp(END_DATE));
	    bean.setAppState(ApplicationState.toApplicationState(rs.getInt(APP_STATE)));
	    
		return bean;
	}

}
