/* $Id: MySQLActivityHistoryDAO.java,v 1.1 2007/10/29 02:55:36 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.mysql;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.ActivityHistoryBean;
import jp.co.argo21.nautica.workflow.dataaccess.ActivityHistoryDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * ActivityHistoryBeanをデータとするMySQL用DAOクラスである。
 * <pre><tt>
 * ACTIVITYHISTテーブル
 * +-------------+--------------+------+-----+---------------------+-------+
 * | Field       | Type         | Null | Key | Default             | Extra |
 * +-------------+--------------+------+-----+---------------------+-------+
 * | PROC_ID     | varchar(64)  |      | PRI |                     |       |
 * | ACT_ID      | varchar(64)  |      | PRI |                     |       |
 * | ACT_DEF_ID  | varchar(64)  |      |     |                     |       |
 * | ACTOR_ID    | varchar(64)  |      |     |                     |       |
 * | PREV_STATE  | int(1)       | YES  |     | NULL                |       |
 * | NEW_STATE   | int(1)       |      |     | 0                   |       |
 * | UPDATE_DATE | datetime     |      |     | 0000-00-00 00:00:00 |       |
 * | REASON      | varchar(128) | YES  |     | NULL                |       |
 * +-------------+--------------+------+-----+---------------------+-------+
 * </tt></pre>
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class MySQLActivityHistoryDAO extends AbstractDAO implements ActivityHistoryDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/**　プロセスIDのカラム名 */
	public static final String PROC_ID     = "PROC_ID";
	/**　アクティビティIDのカラム名 */
	public static final String ACT_ID      = "ACT_ID";
	/**　アクティビティ定義IDのカラム名 */
	public static final String ACT_DEF_ID  = "ACT_DEF_ID";
	/**　アクティビティIDのカラム名 */
	public static final String ACTOR_ID    = "ACTOR_ID";
	/**　前回の状態のカラム名 */
	public static final String PREV_STATE  = "PREV_STATE";
	/**　新しい状態のカラム名 */
	public static final String NEW_STATE   = "NEW_STATE";
	/**　更新日時のカラム名 */
	public static final String UPDATE_DATE = "UPDATE_DATE";
	/**　更新事由のカラム名 */
	public static final String REASON      = "REASON";

	private static final String SQL_INSERT
	= "insert into ACTIVITYHIST ("
		+ PROC_ID + ", "
		+ ACT_ID + ", "
		+ ACT_DEF_ID + ", "
		+ ACTOR_ID + ", "
		+ PREV_STATE + ", "
		+ NEW_STATE + ", "
		+ UPDATE_DATE + ", "
		+ REASON + ") values (?,?,?,?,?,?,?,?)";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	MySQLActivityHistoryDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * アクティビティ変更履歴を挿入する。
	 *
	 * @param bean アクティビティ変更履歴
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.ActivityHistoryDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.ActivityHistoryBean)
	 */
	public void insert(ActivityHistoryBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getProcID());
			st.setString(2, bean.getActID());
			st.setString(3, bean.getActDefID());
			st.setString(4, bean.getActorID());
			st.setInt(5, bean.getPrevState());
			st.setInt(6, bean.getNewState());
			st.setTimestamp(7, bean.getUpdateDate());
			st.setString(8, bean.getReason());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// アクティビティ履歴レコードの挿入に失敗しました。
				String errMsg = StringManager.get("E0012");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットからアクティビティ変更履歴を作成する。
	 *
	 * @param rs 結果セット
	 * @return アクティビティ変更履歴
	 * @throws SQLException 任意のSQL例外
	 */
	@SuppressWarnings("unused")
	private ActivityHistoryBean createBean(ResultSet rs)
	throws SQLException
	{
		ActivityHistoryBean bean = new ActivityHistoryBean();

		bean.setProcID(rs.getString(PROC_ID));
		bean.setActID(rs.getString(ACT_ID));
		bean.setActDefID(rs.getString(ACT_DEF_ID));
		bean.setActorID(rs.getString(ACTOR_ID));
		bean.setPrevState(rs.getInt(PREV_STATE));
		bean.setNewState(rs.getInt(NEW_STATE));
		bean.setUpdateDate(rs.getTimestamp(UPDATE_DATE));
		bean.setReason(rs.getString(REASON));
		
		return bean;
	}
}
