/* $Id: DerbyVariableHistoryDAO.java,v 1.3 2007/11/27 02:06:21 knakata Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.VariableHistoryBean;
import jp.co.argo21.nautica.workflow.dataaccess.VariableHistoryDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.util.StringManager;

/**
 * VariableHistoryBeanをデータとするDerby用DAOクラスである。
 * <pre><tt>
 * WF_VARIABLEHISTテーブル
 * +-------------+--------------+------+-----+---------------------+-------+
 * | Field       | Type         | Null | Key | Default             | Extra |
 * +-------------+--------------+------+-----+---------------------+-------+
 * | VAR_ID      | varchar(64)  |      | PRI |                     |       |
 * | PROC_ID     | varchar(64)  |      |     |                     |       |
 * | VAR_DEF_ID  | varchar(64)  |      |     |                     |       |
 * | TYPE        | int(1)       |      |     | 0                   |       |
 * | PREV_VALUE  | text         | YES  |     | NULL                |       |
 * | NEW_VALUE   | text         |      |     |                     |       |
 * | UPDATE_DATE | datetime     |      |     | 0000-00-00 00:00:00 |       |
 * | REASON      | varchar(128) | YES  |     | NULL                |       |
 * | ACT_ID      | varchar(64)  | YES  |     | NULL                |       |
 * +-------------+--------------+------+-----+---------------------+-------+
 * </tt></pre>
 *
 * @author  knakata(Argo 21, Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class DerbyVariableHistoryDAO extends AbstractDAO implements VariableHistoryDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** プロセス変数IDのカラム名 */
	public static final String VAR_ID      = "VAR_ID";
	/** プロセスIDのカラム名 */
	public static final String PROC_ID     = "PROC_ID";
	/** プロセス変数定義IDのカラム名 */
	public static final String VAR_DEF_ID  = "VAR_DEF_ID";
	/** プロセス変数タイプのカラム名 */
	public static final String TYPE        = "TYPE";
	/** 前回の値のカラム名 */
	public static final String PREV_VALUE  = "PREV_VALUE";
	/** 新しい値のカラム名 */
	public static final String NEW_VALUE   = "NEW_VALUE";
	/** 更新日時のカラム名 */
	public static final String UPDATE_DATE = "UPDATE_DATE";
	/** 更新事由のカラム名 */
	public static final String REASON      = "REASON";
	/** アクティビティIDのカラム名 */
	public static final String ACT_ID      = "ACT_ID";

	private static final String SQL_INSERT
	= "insert into WF_VARIABLEHIST ("
		+ VAR_ID + ", "
		+ PROC_ID + ", "
		+ VAR_DEF_ID + ", "
		+ TYPE + ", "
		+ PREV_VALUE + ", "
		+ NEW_VALUE + ", "
		+ UPDATE_DATE + ", "
		+ REASON + ", "
		+ ACT_ID + ") values (?,?,?,?,?,?,?,?,?)";

	/**
	 * プロセス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	DerbyVariableHistoryDAO(Connection conn)
	{
		super(conn);
	}
		
	/**
	 * プロセス変数変更履歴を挿入する。
	 *
	 * @param bean プロセス変数変更履歴
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.VariableHistoryDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.VariableHistoryBean)
	 */
	public void insert(VariableHistoryBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getVarID());
			st.setString(2, bean.getProcID());
			st.setString(3, bean.getVarDefID());
			st.setInt(4, bean.getType());
			st.setString(5, bean.getPrevValue());
			st.setString(6, bean.getNewValue());
			st.setTimestamp(7, bean.getUpdateDate());
			st.setString(8, bean.getReason());
			st.setString(9, bean.getActivityID());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// プロセス変数変更履歴の挿入に失敗しました。
				String errMsg = StringManager.get("E0033");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットからプロセス変数変更履歴を作成する。
	 *
	 * @param rs 結果セット
	 * @return プロセス変数変更履歴
	 * @throws SQLException 任意のSQL例外
	 */
	@SuppressWarnings("unused")
	private VariableHistoryBean createBean(ResultSet rs)
	throws SQLException
	{
		VariableHistoryBean bean = new VariableHistoryBean();

		bean.setVarID(rs.getString(VAR_ID));
		bean.setProcID(rs.getString(PROC_ID));
		bean.setVarDefID(rs.getString(VAR_DEF_ID));
		bean.setType(rs.getInt(TYPE));
		bean.setPrevValue(rs.getString(PREV_VALUE));
		bean.setNewValue(rs.getString(NEW_VALUE));
		bean.setUpdateDate(rs.getTimestamp(UPDATE_DATE));
		bean.setReason(rs.getString(REASON));
		bean.setActivityID(rs.getString(ACT_ID));
		
		return bean;
	}
}
