/* $Id: DerbySequencerDAO.java,v 1.5 2007/11/27 02:06:21 knakata Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.SequencerDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;

/**
 * ユニークIDを生成するためのシーケンサを提供するクラスである。
 *
 * @author  knakata(Argo 21, Corp.)
 * @version $Revision: 1.5 $
 * @since   Nautica Workflow 1.0
 */
public class DerbySequencerDAO extends AbstractDAO implements SequencerDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** シーケンス番号のカラム名 */
	public static final String SEQ  = "SEQ";
	/** シーケンス名のカラム名 */
	public static final String NAME = "NAME";

	/** 新しいシーケンサを追加するためのSQL文 */
	private static final String SQL_INSERT_SEQUENCER
		= "INSERT INTO WF_KEYSEQ (NAME, SEQ) VALUES (?, 1) ";
	/** シーケンサの値を取得するためのSQL文 */
	private static final String SQL_GET_SEQUENCER
		= "SELECT SEQ FROM WF_KEYSEQ WHERE NAME = ? ";
	/** シーケンサの値を更新するためのSQL文 */
	private static final String SQL_UPDATE_SEQUENCER
		= "UPDATE WF_KEYSEQ SET SEQ = SEQ + 1 WHERE NAME = ? ";
	/**	行ロックを獲得するためのSQL文 */
	private static final String SQL_LOCK_SEQUENCER
		= "SELECT * FROM WF_KEYSEQ WHERE NAME = ? FOR UPDATE ";

	/**
	 * シーケンス管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	DerbySequencerDAO(Connection conn)
	{
		super(conn);
	}

	/**
	 * 次のシーケンス値をintの範囲内で取得する。
	 *
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	public int nextForInt(String seqName) throws SQLException
	{
		long seq = next(seqName);
		if (seq > Integer.MAX_VALUE) {
			throw new ArithmeticException("Value is out of INTEGER range.");
		}
		return (int)seq;
	}

	/**
	 * 次のシーケンス値を取得する。
	 *
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	public synchronized long next(String seqName) throws SQLException 
	{
		Connection conn = getConnection();
		conn.setAutoCommit(false);
		PreparedStatement statement = conn.prepareStatement(SQL_LOCK_SEQUENCER);
		try {
			statement.setString(1, seqName);
			ResultSet rs = statement.executeQuery();
			if (rs.next()) {
				updateValue(conn, seqName);
			} else {
				insertValue(conn, seqName);
			}
			
			long seq = currentValue(conn, seqName);
			conn.commit();
			return seq;
		} catch (SQLException e) {
			conn.rollback();
			throw new SQLException(e);
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}

	/**
	 * シーケンス値を更新する。
	 *
	 * @param conn    JDBC接続
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	public void updateValue(Connection conn, String seqName) throws SQLException
	{
		PreparedStatement statement = conn.prepareStatement(SQL_UPDATE_SEQUENCER);
		try {
			statement.setString(1, seqName);

			/************** DEBUG **************/
			eLog.debug(statement.toString());
			/***********************************/
			statement.executeUpdate();
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}

	/**
	 * 現在のシーケンス値を取得する。
	 *
	 * @param conn    JDBC接続
	 * @param seqName シーケンサ名
	 * @return シーケンス値
	 * @throws SQLException SQL例外
	 */
	private long currentValue(Connection conn, String seqName)
	throws SQLException
	{
		PreparedStatement statement = conn.prepareStatement(SQL_GET_SEQUENCER);
		try {
			statement.setString(1, seqName);

			/************** DEBUG **************/
			eLog.debug(statement.toString());
			/***********************************/

			long seq = -1L;
			ResultSet rs = statement.executeQuery();
			if (rs.next()) {
				seq = rs.getLong(SEQ);
			}
			rs.close();

			return seq;
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}
	
	/**
	 * 新しいシーケンス値を設定する。
	 * @param conn    JDBC接続
	 * @param seqName シーケンサ名
	 * @throws SQLException SQL例外
	 */
	public void insertValue(Connection conn, String seqName) throws SQLException
	{
		PreparedStatement statement = conn.prepareStatement(SQL_INSERT_SEQUENCER);
		try {
			statement.setString(1, seqName);

			/************** DEBUG **************/
			eLog.debug(statement.toString());
			/***********************************/
			statement.executeUpdate();
		} finally {
			if (statement != null) {
				statement.close();
			}
		}
	}
}
