/* $Id: DerbyDefinitionDAO.java,v 1.3 2007/11/27 02:06:21 knakata Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;

import jp.co.argo21.nautica.workflow.dataaccess.AbstractDAO;
import jp.co.argo21.nautica.workflow.dataaccess.DAOException;
import jp.co.argo21.nautica.workflow.dataaccess.DefinitionBean;
import jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO;
import jp.co.argo21.nautica.workflow.engine.LogManager;
import jp.co.argo21.nautica.workflow.filter.FilterConverter;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * DefinitionBeanをデータとするDerby用DAOクラスである。
 * <pre><tt>
 * WF_DEFINITIONテーブル
 * +---------------+--------------+------+-----+---------------------+-------+
 * | Field         | Type         | Null | Key | Default             | Extra |
 * +---------------+--------------+------+-----+---------------------+-------+
 * | PROC_DEF_ID   | varchar(64)  |      | PRI |                     |       |
 * | PATH          | varchar(128) |      |     |                     |       |
 * | INTERNAL_PDID | varchar(64)  |      |     |                     |       |
 * | REGISTER_DATE | datetime     |      |     | 0000-00-00 00:00:00 |       |
 * | DEF_STATE     | int(1)       |      |     | 0                   |       |
 * +---------------+--------------+------+-----+---------------------+-------+
 * </tt></pre>
 *
 * @author  knakata(Argo 21, Corp.)
 * @version $Revision: 1.3 $
 * @since   Nautica Workflow 1.0
 */
public class DerbyDefinitionDAO extends AbstractDAO
implements DefinitionDAO
{
	/** エンジンログ */
	protected static Logger eLog = LogManager.getEngineLogger();

	/** プロセス定義IDのカラム名 */
	public static final String PROC_DEF_ID   = "PROC_DEF_ID";
	/** 定義ファイルのパスのカラム名 */
	public static final String PATH          = "PATH";
	/** 内部プロセス定義IDのカラム名 */
	public static final String INTERNAL_PDID = "INTERNAL_PDID";
	/** 登録日時のカラム名 */
	public static final String REGISTER_DATE = "REGISTER_DATE";
	/** 定義状態のカラム名 */
	public static final String DEF_STATE     = "DEF_STATE";

	private static final String SQL_SELECT_BY_PRIMARY_KEY
	= "select * from WF_DEFINITION where PROC_DEF_ID = ? ";

	private static final String SQL_SELECT_BY_PATH_AND_ID
	= "select * from WF_DEFINITION where PATH = ? and INTERNAL_PDID = ? ";

	private static final String SQL_SELECT
	= "select * from WF_DEFINITION ";

	private static final String SQL_INSERT
	= "insert into WF_DEFINITION ("
		+ PROC_DEF_ID + ", "
		+ PATH + ", "
		+ INTERNAL_PDID + ", "
		+ REGISTER_DATE + ", "
		+ DEF_STATE + ") values (?,?,?,?,?)";

	private static final String SQL_UPDATE_STATE
	= "update WF_DEFINITION set "
		+ DEF_STATE + " = ? where "
		+ PROC_DEF_ID + " = ? AND "+ DEF_STATE + " = ? ";

	/**
	 * 定義管理テーブルに対するアクセスオブジェクトを生成する。
	 *
	 * @param conn コネクション
	 */
	DerbyDefinitionDAO(Connection conn)
	{
		super(conn);
	}
	
	/**
	 * 主キーにより、定義情報を取得する。
	 *
	 * @param pdid 定義ID
	 * @return 定義情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO#findByPrimaryKey(java.lang.String)
	 */
	public DefinitionBean findByPrimaryKey(String pdid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PRIMARY_KEY);
		try {
			st.setString(1, pdid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			DefinitionBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}

    /**
	 * パスと内部プロセス定義IDで定義情報を取得する。
	 *
	 * @param path 定義ファイルへの相対パス
     * @param ipdid 内部プロセス定義ID
	 * @return 定義情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
     * @see jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO#findByPathAndID(java.lang.String, java.lang.String)
     */
	public DefinitionBean findByPathAndID(String path, String ipdid)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT_BY_PATH_AND_ID);
		try {
			st.setString(1, path);
			st.setString(2, ipdid);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			ResultSet rs = st.executeQuery();
			
			DefinitionBean bean = null;
			if (rs.next()) {
				bean = createBean(rs);
			}
			return bean;
		} finally {
			st.close();
		}
	}
	
	/**
	 * フィルターにより特定された定義情報を取得する。
	 *
	 * @param filter フィルター
	 * @return 定義情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 無効なフィルターが指定された場合
	 * @see jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO#findByFilter(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public List<DefinitionBean> findByFilter(Filter filter)
	throws SQLException, DAOException, InvalidFilterException
	{
		FilterConverter converter = new DefinitionFilterConverter();
		String where = converter.convertToSQL(filter);
		
		if (where.equals("") == false) {
			where = " where " + where;
		}
		PreparedStatement st = getConnection().prepareStatement(SQL_SELECT + where);

		/************** DEBUG **************/
		eLog.debug(st.toString());
		/***********************************/

		try {
			ResultSet rs = st.executeQuery();
			
			List<DefinitionBean> list = new ArrayList<DefinitionBean>();
			while (rs.next()) {
				DefinitionBean bean = createBean(rs);
				list.add(bean);
			}
			return list;
		} finally {
			st.close();
		}
	}
	
	/**
	 * 定義情報を挿入する。
	 *
	 * @param bean 定義情報
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO#insert(jp.co.argo21.nautica.workflow.dataaccess.DefinitionBean)
	 */
	public void insert(DefinitionBean bean)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_INSERT);
		try {
			st.setString(1, bean.getProcessDefinitionID());
			st.setString(2, bean.getPath());
			st.setString(3, bean.getInternalPDID());
			st.setTimestamp(4, bean.getRegisterDate());
			st.setInt(5, bean.getState());

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 定義情報レコードの挿入に失敗しました。
				String errMsg = StringManager.get("E0016");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 定義情報の状態を更新する。
	 *
	 * @param pdid 定義ID
	 * @param oldState 前の状態
	 * @param newState 後の状態
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @see jp.co.argo21.nautica.workflow.dataaccess.DefinitionDAO#updateState(java.lang.String, int, int)
	 */
	public void updateState(String pdid, int oldState, int newState)
	throws SQLException, DAOException
	{
		PreparedStatement st = getConnection().prepareStatement(SQL_UPDATE_STATE);
		try {
			st.setInt(1, newState);
			st.setString(2, pdid);
			st.setInt(3, oldState);

			/************** DEBUG **************/
			eLog.debug(st.toString());
			/***********************************/
			
			int i = st.executeUpdate();
			
			if (i != 1) {
				// 定義情報レコードの更新に失敗しました。
				String errMsg = StringManager.get("E0017");
				throw new DAOException(errMsg);
			}
		} finally {
			st.close();
		}
	}
	
	/**
	 * 結果セットから定義情報を作成する。
	 *
	 * @param rs 結果セット
	 * @return 定義情報
	 * @throws SQLException 任意のSQL例外
	 */
	private DefinitionBean createBean(ResultSet rs)
	throws SQLException
	{
		DefinitionBean bean = new DefinitionBean();

		bean.setProcessDefinitionID(rs.getString(PROC_DEF_ID));
		bean.setPath(rs.getString(PATH));
		bean.setInternalPDID(rs.getString(INTERNAL_PDID));
		bean.setRegisterDate(rs.getTimestamp(REGISTER_DATE));
		bean.setState(rs.getInt(DEF_STATE));
		
		return bean;
	}
}
