/* $Id: DefinitionFilterConverter.java,v 1.2 2007/11/28 04:16:42 knakata Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess.derby;

import jp.co.argo21.nautica.workflow.filter.FilterConverter;
import jp.co.argo21.nautica.workflow.filter.GroupFilter;
import jp.co.argo21.nautica.workflow.filter.NameFilter;
import jp.co.argo21.nautica.workflow.filter.StateFilter;
import jp.co.argo21.nautica.workflow.util.StringManager;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;

/**
 * 定義検索用のフィルターからSQLのwhere句を生成するコンバータ。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class DefinitionFilterConverter implements FilterConverter
{
    /**
     * コンバータを生成する。
     *
     */
    public DefinitionFilterConverter() {
        // 処理なし。
    }
    
	/**
	 * 定義検索用のフィルターからSQLのwhere句を生成する
	 *
	 * @param filter フィルター
	 * @return SQLのwhere句
	 * @see jp.co.argo21.nautica.workflow.filter.FilterConverter#convertToSQL(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public String convertToSQL(Filter filter) throws InvalidFilterException
	{
		if (filter == null) {
			return "";
		} else if (filter instanceof NameFilter) {
			return convertToSQL((NameFilter)filter);
		} else if (filter instanceof StateFilter) {
			return convertToSQL((StateFilter)filter);
		} else if (filter instanceof GroupFilter) {
			return convertToSQL((GroupFilter)filter);
		} else {
			// 無効なフィルターが指定されました。
			String errMsg = StringManager.get("E0004");
			throw new InvalidFilterException(errMsg);
		}
	}
	
	/**
	 * NameFilterをSQL部分文字列に変換する。
	 *
	 * @param filter NameFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(NameFilter filter)
	throws InvalidFilterException
	{
		return DerbyDefinitionDAO.PROC_DEF_ID 
			+ getComparisonString(filter.getComparison())
			+ "'" + filter.getFilterString() + "'";
	}
	
	/**
	 * StateFilterをSQL部分文字列に変換する。
	 *
	 * @param filter StateFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(StateFilter filter)
	throws InvalidFilterException
	{
		return DerbyDefinitionDAO.DEF_STATE
			+ " = " +  filter.getFilterString();
	}
	
	/**
	 * GroupFilterをSQL部分文字列に変換する。
	 *
	 * @param filter GroupFilter
	 * @return SQL部分文字列
	 * @throws InvalidFilterException 利用できないフィルターの場合
	 */
	private String convertToSQL(GroupFilter filter)
	throws InvalidFilterException
	{
		Filter[] children = filter.getChild();
		
		int count = children.length;
		if (count == 0) return "";
		
		String s = convertToSQL(children[0]);
		
		for (int i = 1; i < count; i++) {
			s += " AND " + convertToSQL(children[i]);
		}
		
		return s;
	}
	
	/**
	 * 演算子を文字列に変換する。
	 *
	 * @param comparison 演算子
	 * @return 部分文字列
	 */
	private String getComparisonString(int comparison)
	{
		switch (comparison) {
		case NameFilter.EQUAL:
			return " = ";
		case NameFilter.GREATER_EQUAL:
			return " >= ";
		case NameFilter.GREATER_THAN:
			return " > ";
		case NameFilter.LESS_EQUAL:
			return " <= ";
		case NameFilter.LESS_THAN:
			return " < ";
		case NameFilter.NOT_EQUAL:
			return " <> ";
		case NameFilter.LIKE:
			return " LIKE ";
		default:
			return " <> ";
		}
	}
}
