/* $Id: WorkItemDAO.java,v 1.2 2007/12/05 05:56:17 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess;

import java.sql.SQLException;
import java.util.List;

import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.InvalidFilterException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidQueryException;
import jp.co.argo21.nautica.workflow.wfmc.InvalidWorkItemException;
import jp.co.argo21.nautica.workflow.wfmc.Participant;
import jp.co.argo21.nautica.workflow.wfmc.WorkItem;

/**
 * WorkItemBeanをデータとするDAOインタフェース。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public interface WorkItemDAO
{
	/**
	 * 作業項目IDで作業項目を取得する。
	 *
	 * @param wid 作業項目ID
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public WorkItemBean findByPrimaryKey(String wid)
	throws SQLException, DAOException;

	/**
	 * プロセスIDとアクティビティIDで作業項目を取得する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public WorkItemBean findByProcessIDANDActivityID(String pid, String aid)
	throws SQLException, DAOException;


	/**
	 * ワークフロー参加者名と型で作業項目の個数を取得する。
	 *
	 * @param name ワークフロー参加者名
	 * @param type ワークフロー参加者型
	 * @return 作業項目の個数
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public int findCountByParticipant(String name, int type)
	throws SQLException, DAOException;

	/**
	 * ワークフロー参加者名と型で作業項目を取得する。
	 *
	 * @param name ワークフロー参加者名
	 * @param type ワークフロー参加者型
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public WorkItemBean[] findByParticipant(String name, int type)
	throws SQLException, DAOException;

	/**
	 * SQL部分文字列にしたがって、作業項目を取得する。
	 *
	 * @param s SQL部分文字列
	 * @param hasAttributeFilter 属性フィルターを含む場合true
	 * @return 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public WorkItemBean[] findBySQLPart(String s, boolean hasAttributeFilter)
	throws SQLException, DAOException;

	/**
	 * 作業項目一覧の取得を行う。
	 *
	 * @param filter 検索用フィルター
	 * @param userResource ユーザ情報
	 * @param orgResources 組織情報
	 * @param roleResources ロール情報
	 * @return 作業項目一覧
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidFilterException 指定されたフィルターが有効でない場合
	 * @throws InvalidQueryException SQL部分文字列の生成に失敗した場合
	 */
	public WorkItem[] findByFilter(Filter filter, Participant userResource, List<Participant> orgResources, List<Participant> roleResources)
	throws SQLException, DAOException, InvalidFilterException, InvalidQueryException;

	/**
	 * 対応する作業項目のチェックを行い、それを返す。
	 *
	 * @param wid 作業項目ID
	 * @param userResource ユーザ情報
	 * @param orgResources 組織情報
	 * @param roleResources ロール情報
	 * @return　作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 * @throws InvalidWorkItemException 作業項目が特定できない場合
	 * @throws InvalidQueryException SQL部分文字列の生成に失敗した場合
	 */
	public WorkItem findByWorkItemID(String wid, Participant userResource, List<Participant> orgResources, List<Participant> roleResources)
	throws DAOException, SQLException, InvalidWorkItemException, InvalidQueryException;
	
	/**
	 * 作業項目を挿入する。
	 *
	 * @param bean 作業項目
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void insert(WorkItemBean bean)
	throws SQLException, DAOException;

	/**
	 * 作業項目のワークフロー参加者を更新する。
	 *
	 * @param wid 作業項目ID
	 * @param src 更新前のワークフロー参加者
	 * @param target 更新後のワークフロー参加者
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateParticipant(String wid,Participant src, Participant target)
	throws SQLException, DAOException;

	/**
	 * 作業項目を削除する。
	 *
	 * @param wid 作業項目ID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void delete(String wid)
	throws SQLException, DAOException;

	/**
	 * プロセスIDとアクティビティIDで作業項目を取得する。
	 *
	 * @param pid プロセスID
	 * @param aid アクティビティID
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void deleteByProcessIDANDActivityID(String pid, String aid)
	throws SQLException, DAOException;
}
