/* $Id: AppExecutionDAO.java,v 1.2 2007/11/05 09:50:54 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow Core.
 * 
 *  Nautica Workflow Core is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation; either version 2.1 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow Core is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 * 
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with Nautica Workflow Core; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.workflow.dataaccess;

import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.List;

import jp.co.argo21.nautica.workflow.ta.ApplicationState;

/**
 * アプリケーション実行テーブルに対するDAOインタフェース
 *
 * @author  kiyoda(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public interface AppExecutionDAO 
{
    /**
     * 主キーで、アプリケーション実行テーブルオブジェクトを取得する。
     * 
     * アプリケーション実行テーブルの主キーである、
     * アプリケーション名、ツールエージェントのクラス名、実行IDで、
     * アプリケーション実行テーブルオブジェクトを取得する。
     *
     * @param appName		アプリケーション名
     * @param toolAgent		アプリケーションを起動したツールエージェントのクラス名
     * @param execID		ツールエージェントが発行した実行ID
     * @return		アプリケーション実行テーブルオブジェクト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
     */
    public AppExecutionBean findByPrimaryKey(String appName, String toolAgent, int execID)
	throws SQLException, DAOException;

	/**
	 * 実行IDで、アプリケーション実行テーブルオブジェクトを取得する。
	 * 
	 * 主キーの一部である、実行IDでアプリケーション実行テーブルオブジェクトのリストを
	 * 取得する
	 *
	 * @param execID	実行ID
	 * @return		アプリケーション実行テーブルオブジェクトのリスト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public List<AppExecutionBean> findByExecID(int execID)
	throws SQLException, DAOException;
    
	/**
	 * 開始者IDで、アプリケーション実行テーブルオブジェクトを取得する。
	 * 開始者IDとは、対応する作業項目IDである。
	 *
	 * @param starterID 開始者ID
	 * @return		アプリケーション実行テーブルオブジェクトのリスト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public List<AppExecutionBean> findByStarterID(String starterID)
	throws SQLException, DAOException;
    
	/**
	 * アプリケーション実行テーブルにレコードを追加する。
	 * 
	 * @param bean		アプリケーション実行テーブルオブジェクト
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void insert(AppExecutionBean bean)
	throws SQLException, DAOException;

	/**
	 * 実行状態を更新する
	 *
	 * @param appName		アプリケーション名
	 * @param toolAgent		ツールエージェントのクラス名
	 * @param execID		実行ID
	 * @param oldState		旧実行状態
	 * @param newState		新実行状態
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateState(String appName, String toolAgent, int execID, 
	        ApplicationState oldState, ApplicationState newState)
	throws SQLException, DAOException;

	/**
	 * 実行状態と終了日時を更新する
	 *
	 * @param appName		アプリケーション名
	 * @param toolAgent		ツールエージェントのクラス名
	 * @param execID		実行ID
	 * @param oldState		旧実行状態
	 * @param newState		新実行状態
	 * @param newDate		新終了日時
	 * @throws SQLException 任意のSQL例外
	 * @throws DAOException DAO実行中に処理としてあげる場合がある例外
	 */
	public void updateState(String appName, String toolAgent, int execID, 
	        ApplicationState oldState, ApplicationState newState, Timestamp newDate)
	throws SQLException, DAOException;

}
