/* $Id: ProcessDefinitionFlowView.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.SystemColor;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.AffineTransform;
import java.awt.geom.GeneralPath;
import java.util.List;

import javax.swing.Icon;
import javax.swing.JComponent;

import jp.co.argo21.nautica.commons.swing.Grid;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.commons.util.StringUtils;
import jp.co.argo21.nautica.tool.wfadm.IconManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.definition.TransitionDefinition;

/**
 * プロセス定義のフロー表示エリアである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessDefinitionFlowView
extends JComponent
implements MouseListener, MouseMotionListener
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** グリッド幅を表す定数 */
	public static final int GRID_SIZE           = 80;
	/** アイコンサイズを表す定数 */
	private static final int MAX_ICON_SIZE      = 48;
	/** 内法を表す定数 */
	private static final int INSET_SIZE         = 9;
	/** 最小論理幅単位数を表す定数 */
	private static final int MIN_LOGICAL_WIDTH  = 10;
	/** 最小論理高さ単位数を表す定数 */
	private static final int MIN_LOGICAL_HEIGHT = 10;

	/** 背景色 */
	private static final Color BACKGROUND        = new Color(255, 255, 230);
	/** 同期レイヤーの背景色 */
	private static final Color LAYER_BACKGROUND  = new Color(230, 255, 255, 64);

	/** 描画に使用するフォント */
	private static final Font VIEW_FONT = new Font("MonoSpaced", Font.PLAIN, 10);

	/** 矢印用描画パス */
	private static GeneralPath ARROW_PATH;

	/** カーソル位置の矩形座標 */
	private Rectangle cursorRect;
	/** 選択中の矩形座標 */
	private Rectangle selectedRect;

	/** 対象となるプロセス */
	private ProcessDefinition proc;

	/** 選択中のActivityDefinition */
	private ActivityDefinition selectedActivity;

	/** 同期階層を管理するリスト */
	private ConcurrentLayerManager root;

	/**
	 * 矢印用描画パスの初期化
	 */
	static
	{
		// 矢印用描画パスの初期化
		ARROW_PATH = new GeneralPath(GeneralPath.WIND_EVEN_ODD);
		ARROW_PATH.moveTo(0, 0);
		ARROW_PATH.lineTo(15, 5);
		ARROW_PATH.lineTo(15, -5);
		ARROW_PATH.closePath();
	}

	/**
	 * プロセス定義のフロー表示エリアを生成する。
	 * 
	 * @param proc 対象となるプロセス定義
	 */
	public ProcessDefinitionFlowView(ProcessDefinition proc)
	{
		this.proc = proc;
		int w = proc.getLogicalWidth();
		int h = proc.getLogicalHeight();

		if (w < MIN_LOGICAL_WIDTH) {
			w = MIN_LOGICAL_WIDTH;
		} 
		if (h < MIN_LOGICAL_HEIGHT) {
			h = MIN_LOGICAL_HEIGHT;
		} 
		
		Dimension size = new Dimension(GRID_SIZE * w + 1, GRID_SIZE * h + 1);
		
		root = new ConcurrentLayerManager();
		root.initConcurrentLayer(proc.getActivities());
			
		setPreferredSize(size);
		setMinimumSize(size);
		addMouseListener(this);
		addMouseMotionListener(this);
	}

	/**
	 * フロー表示エリアを描画する。
	 *
	 * @param g 描画対象グラフィックス
	 */
	public void paintComponent(Graphics g)
	{
		Shape clip = g.getClip();

		Dimension size = this.getSize();

		//描画範囲以外の背景の描画
		g.setColor(Color.gray);
		g.fillRect(0, 0, size.width, size.height);

		Dimension psize = this.getPreferredSize();

		//描画範囲の背景の描画
		g.setColor(BACKGROUND);
		g.fillRect(0, 0, psize.width, psize.height);

		//レイヤーの背景描画
		root.paintLayerBackgroud(g, LAYER_BACKGROUND, GRID_SIZE);

		//選択中のアクティビティの描画
		if (selectedActivity != null) {
			paintActivityBackground(g, Color.pink, selectedActivity);
		}

		//ビューの描画
		paintView(g);
						
		//カーソル矩形の描画
		if (cursorRect != null) {
			g.drawRect(cursorRect.x, cursorRect.y, cursorRect.width, cursorRect.height);
		}

		//レイヤーの枠描画
		root.paintLayerBorder(g, Color.blue, GRID_SIZE);

		//選択中のアクティビティの描画
		if (selectedRect != null) {
			g.setColor(Color.red);
			g.drawRect(selectedRect.x, selectedRect.y, selectedRect.width, selectedRect.height);
		}

		g.setClip(clip);
		g.dispose();
	}

	/**
	 * フロー表示エリアを描画する。
	 *
	 * @param g 描画対象グラフィックス
	 */
	public void paintView(Graphics g)
	{
		Shape clip = g.getClip();

		g.setFont(VIEW_FONT);

		Dimension psize = this.getPreferredSize();

		//描画範囲のグリッドの描画
		Grid.draw(new Rectangle(psize), g, GRID_SIZE);
		g.clipRect(0, 0, psize.width, psize.height);

		g.setColor(Color.black);
			
		//アクティビティの描画
		List<ActivityDefinition> acts = proc.getActivities();
		for (ActivityDefinition act : acts) {
			paintActivity(g, act);
		}
			
		//遷移の描画
		List<TransitionDefinition> trs = proc.getTransitions();
		for (TransitionDefinition tr : trs) {
			paintTransition(g, Color.black, tr.getSource(), tr.getDestination());
		}

		g.setClip(clip);
	}

	/**
	 * 指定されたアクティビティ背景を描画する。
	 * 
	 * @param g   描画対象グラフィックス
	 * @param c   描画色
	 * @param act アクティビティ
	 */
	private void paintActivityBackground(Graphics g, Color c, ActivityDefinition act)
	{
		if (act == null) return;

		int ax = act.getLogicalX();
		int ay = act.getLogicalY();

		int x = GRID_SIZE * ax;
		int y = GRID_SIZE * ay;
		int w = GRID_SIZE;
		int h = GRID_SIZE;

		g.setColor(c);
		g.fillRect(x, y, w, h);
	}

	/**
	 * 指定されたアクティビティを描画する。
	 * 
	 * @param g  描画対象グラフィックス
	 * @param act アクティビティ
	 */
	private void paintActivity(Graphics g, ActivityDefinition act)
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		String name = act.getName();
		if (StringUtils.checkNull(name)) {
			name = rm.getResource("ProcessDefinitionFlowView.noname");
		} 

		//アクティビティアイコンの描画
		String type = act.getType();
		String iconName = getIconName(type);
		Icon icon = IconManager.getIcon(iconName);

		int ax = act.getLogicalX();
		int ay = act.getLogicalY();

		int x = GRID_SIZE * ax;
		int y = GRID_SIZE * ay;
		int w = GRID_SIZE;
		//int h = GRID_SIZE;

		int iconW = icon.getIconWidth();
		int iconH = icon.getIconHeight();
		int iconX = x + ((w - iconW) / 2);
		int iconY = y + 9;
		icon.paintIcon(this, g, iconX, iconY);

		//アクティビティ名の描画
		FontMetrics metrics = g.getFontMetrics();
		int fontH = metrics.getHeight();
		int descent = metrics.getDescent();
		int lineW = metrics.stringWidth(name);
		int dx = x + ((w - lineW) / 2);
		int dy = y + 9 + iconH;
		if (dx < 0) dx = 0;
		dy = dy + fontH - descent;
		g.setColor(SystemColor.textText);
		g.drawString(name, dx, dy);
	}

	/**
	 * 指定されたアクティビティ間の遷移を描画する。
	 * 
	 * @param g   描画対象グラフィックス
	 * @param c   描画色
	 * @param src 遷移元アクティビティ
	 * @param dst 遷移先アクティビティ
	 */
	private void paintTransition(Graphics g, Color c, ActivityDefinition src, ActivityDefinition dst)
	{
		if (src == null) return;
		if (dst == null) return;

		int sx = src.getLogicalX();
		int sy = src.getLogicalY();
		int dx = dst.getLogicalX();
		int dy = dst.getLogicalY();

		Point sp = getSourcePoint(sx, sy, dx, dy);
		Point dp = getDestinationPoint(sx, sy, dx, dy);

		//遷移線の描画
		paintArrowLine(g, c, sp.x, sp.y, dp.x, dp.y);
	}

	/**
	 * 矢印線を描画する。
	 *
	 * @param g  描画対象グラフィックス
	 * @param c  描画色
	 * @param sx 開始X座標
	 * @param sy 開始Y座標
	 * @param dx 終了X座標
	 * @param dy 終了Y座標
	 */
	private void paintArrowLine(Graphics g, Color c, int sx, int sy, int dx, int dy)
	{
		Graphics2D g2 = (Graphics2D)g;

		//現在の座標変換行列を保存
		AffineTransform tr = g2.getTransform();

		g2.setColor(c);
		g2.drawLine(sx, sy, dx, dy);

		//矢印を描画
		g2.translate(dx, dy);
		double theta = Math.atan2(((double)(sy - dy)), ((double)(sx - dx)));
		g2.rotate(theta);
		g2.fill(ARROW_PATH);
		g2.draw(ARROW_PATH);

		g2.setTransform(tr);
	}

	/**
	 * 接続元の描画開始点を算出する。
	 *
	 * @param sx 開始X座標
	 * @param sy 開始Y座標
	 * @param dx 終了X座標
	 * @param dy 終了Y座標
	 * @return 描画開始点
	 */
	private Point getSourcePoint(int sx, int sy, int dx, int dy)
	{
		int x = (sx * GRID_SIZE) + (GRID_SIZE / 2);
		int y = (sy * GRID_SIZE) + (GRID_SIZE / 2);
		if (sx < dx) {
			x = (sx * GRID_SIZE) + (GRID_SIZE - ((GRID_SIZE - MAX_ICON_SIZE) / 2));
			y = (sy * GRID_SIZE) + INSET_SIZE + (MAX_ICON_SIZE / 2);
		} else if (sx > dx) {
			x = (sx * GRID_SIZE) + ((GRID_SIZE - MAX_ICON_SIZE) / 2);
			y = (sy * GRID_SIZE) + INSET_SIZE + (MAX_ICON_SIZE / 2);
		} else {
			if (sy < dy) {
				y = (sy * GRID_SIZE) + GRID_SIZE - INSET_SIZE;
			} else if (sy > dy) {
				y = (sy * GRID_SIZE) + INSET_SIZE;
			}
		}
		return new Point(x, y);
	}

	/**
	 * 接続元の描画終了点を算出する。
	 *
	 * @param sx 開始X座標
	 * @param sy 開始Y座標
	 * @param dx 終了X座標
	 * @param dy 終了Y座標
	 * @return 描画終了点
	 */
	private Point getDestinationPoint(int sx, int sy, int dx, int dy)
	{
		int x = (dx * GRID_SIZE) + (GRID_SIZE / 2);
		int y = (dy * GRID_SIZE) + (GRID_SIZE / 2);
		if (sx < dx) {
			x = (dx * GRID_SIZE) + ((GRID_SIZE - MAX_ICON_SIZE) / 2);
			y = (dy * GRID_SIZE) + INSET_SIZE + (MAX_ICON_SIZE / 2);
		} else if (sx > dx) {
			x = (dx * GRID_SIZE) + (GRID_SIZE - ((GRID_SIZE - MAX_ICON_SIZE) / 2));
			y = (dy * GRID_SIZE) + INSET_SIZE + (MAX_ICON_SIZE / 2);
		} else {
			if (sy < dy) {
				y = (dy * GRID_SIZE) + INSET_SIZE;
			} else if (sy > dy) {
				y = (dy * GRID_SIZE) + GRID_SIZE - INSET_SIZE;
			}
		}
		return new Point(x, y);
	}
	/**
	 * マウスカーソルの座標からアクティビティを特定する。
	 * 
	 * @param mx マウスのX座標
	 * @param my マウスのY座標
	 * @return アクティビティ
	 */
	private ActivityDefinition getActivity(int mx, int my)
	{
		int lx = mx / GRID_SIZE;
		int ly = my / GRID_SIZE;

		List<ActivityDefinition> acts = proc.getActivities();
		for (ActivityDefinition act : acts) {
			int x = act.getLogicalX();
			int y = act.getLogicalY();
			if (x == lx && y == ly) {
				return act;
			}
		}
		return null;
	}

	/**
	 * 指定されたアクティビティを選択状態にする。
	 * 
	 * @param act	アクティビティ
	 */
	public void setSelectedActivity(ActivityDefinition act)
	{
		selectedActivity = act;
	}
	
	/**
	 * アクティビティ型からアイコン名を返す。
	 *
	 * @param type 型名
	 * @return アイコン名
	 */
	private String getIconName(String type)
	{
		String iconName = null;
		
		if (type.equals(ActivityDefinition.START))             iconName = "start";
		else if (type.equals(ActivityDefinition.INTERACTIVE))  iconName = "interactive";
		else if (type.equals(ActivityDefinition.AUTOMATIC))    iconName = "automatic";
		else if (type.equals(ActivityDefinition.SUBPROCESS))   iconName = "subprocess";
		else if (type.equals(ActivityDefinition.CONCUR_START)) iconName = "concurrent-start";
		else if (type.equals(ActivityDefinition.CONCUR_END))   iconName = "concurrent-end";
		else if (type.equals(ActivityDefinition.END))          iconName = "end";
		
		return iconName;
	}

	/**
	 * 選択されているアクティビティを返す。
	 * 
	 * @return	アクティビティ
	 */
	public ActivityDefinition getSelectedActivity()
	{
		return selectedActivity;
	}

	/**
	 * マウスクリックを判定する。
	 *
	 * @param e マウスイベント
	 * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
	 */
	public void mouseClicked(MouseEvent e)
	{
		int mx = e.getX();
		int my = e.getY();
		int lx = mx / GRID_SIZE;
		int ly = my / GRID_SIZE;

		if (selectedRect == null) {
			selectedRect = new Rectangle(GRID_SIZE, GRID_SIZE);
		} 
		selectedRect.x = lx * GRID_SIZE;
		selectedRect.y = ly * GRID_SIZE;
		
		setSelectedActivity(getActivity(mx, my));

		repaint();
	}

	/**
	 * マウス押下を判定する。
	 *
	 * @param e マウスイベント
	 * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
	 */
	public void mousePressed(MouseEvent e)
	{
	}

	/**
	 * マウスの押下解除を判定する。
	 *
	 * @param e マウスイベント
	 * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
	 */
	public void mouseReleased(MouseEvent e)
	{
	}

	/**
	 * マウスの侵入を判定する。
	 *
	 * @param e マウスイベント
	 * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
	 */
	public void mouseEntered(MouseEvent e)
	{
	}

	/**
	 * マウスの脱出を判定する。
	 *
	 * @param e マウスイベント
	 * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
	 */
	public void mouseExited(MouseEvent e)
	{
		cursorRect = null;
		repaint();
	}

	/**
	 * マウスドラッグを判定する。
	 *
	 * @param	e		マウスイベント
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 */
	public void mouseDragged(MouseEvent e)
	{
		mouseMoved(e);
	}

	/**
	 * マウス移動を判定する。
	 *
	 * @param	e		マウスイベント
	 * @see java.awt.event.MouseMotionListener#mouseMoved(java.awt.event.MouseEvent)
	 */
	public void mouseMoved(MouseEvent e)
	{
		int mx = e.getX();
		int my = e.getY();

		Dimension psize = ProcessDefinitionFlowView.this.getPreferredSize();

		if ((mx >= psize.width) || (my >= psize.height)) {
			cursorRect = null;
			repaint();
			return;
		}
		int lx = mx / GRID_SIZE;
		int ly = my / GRID_SIZE;
				
		if (cursorRect == null) {
			cursorRect = new Rectangle(GRID_SIZE, GRID_SIZE);
		} 
		cursorRect.x = lx * GRID_SIZE;
		cursorRect.y = ly * GRID_SIZE;

		repaint();
	}
}
