/* $Id: PreferenceComponent.java,v 1.1 2007/10/19 07:29:57 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.pref;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.border.Border;
import javax.swing.table.AbstractTableModel;

import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.DialogComponent;

/**
 * ユーザ環境設定用画面コンテンツ
 * 
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.1 $
 * @since   Nautica Workflow 1.0
 */
public class PreferenceComponent
extends DialogComponent
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** テーブルの列名 */
	private static String[] columnNames;
	/** 枠線 */
	protected static final Border border = BorderFactory.createEtchedBorder();

	/** エンジン情報テーブル */
	private JTable engineInfoTable;
	/** エンジン情報追加ボタン */
	private JButton addInfoButton;
	/** エンジン情報編集ボタン */
	private JButton editInfoButton;
	/** エンジン情報削除ボタン */
	private JButton delInfoButton;
	
	/** エンジン情報編集画面 */
	private EngineInfoComponent engineInfoComponent;

	/**
	 * 定数の初期化
	 */
	static
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		columnNames = new String[4];
		columnNames[0] = rm.getResource("PreferenceComponent.columnName.0");
		columnNames[1] = rm.getResource("PreferenceComponent.columnName.1");
		columnNames[2] = rm.getResource("PreferenceComponent.columnName.2");
		columnNames[3] = rm.getResource("PreferenceComponent.columnName.3");
	}

	/**
	 * ユーザ環境設定用画面を生成する。
	 */
	public PreferenceComponent()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		engineInfoComponent = new EngineInfoComponent();

		JPanel panel = createEngineInfoPanel(rm);
		addTab(rm.getResource("PreferenceComponent.tab.0"), panel);

		validate();
	}
	
	/**
	 * エンジン情報一覧のパネルを生成する。
	 * 
	 * @param rm リソースマネージャ
	 * @return パネル
	 */
	private JPanel createEngineInfoPanel(ResourceManager rm)
	{
		JPanel panel = new JPanel();
		panel.setLayout(new BorderLayout());
		panel.setBorder(border);
		Dimension size = new Dimension(500, 365);
		panel.setSize(size);
		panel.setPreferredSize(size);

		engineInfoTable = new JTable(new EngineInfoTableModel());
		engineInfoTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		JScrollPane sp = new JScrollPane();
		sp.setViewportView(engineInfoTable);
		sp.setBorder(BorderFactory.createEmptyBorder(10, 10, 0, 10));

		addInfoButton = new JButton(rm.getResource("JButton.label.add"));
		addInfoButton.addActionListener(new AddInfoAction());
		editInfoButton = new JButton(rm.getResource("JButton.label.edit"));
		editInfoButton.addActionListener(new EditInfoAction());
		delInfoButton = new JButton(rm.getResource("JButton.label.delete"));
		delInfoButton.addActionListener(new DeleteInfoAction());

		JPanel buttonPane = new JPanel();
		buttonPane.setLayout(new FlowLayout(FlowLayout.RIGHT, 10, 10));
		buttonPane.add(addInfoButton);
		buttonPane.add(editInfoButton);
		buttonPane.add(delInfoButton);
		
		panel.add(sp, BorderLayout.CENTER);
		panel.add(buttonPane, BorderLayout.SOUTH);

		validate();
		
		return panel;
	}

	/**
	 * インスペクタを初期化する。
	 * 
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#init()
	 */
	public void init()
	{
	}

	/**
	 * 
	 *
	 * @return
	 * @see jp.co.argo21.nautica.tool.wfadm.common.DialogComponent#commit()
	 */
	public boolean commit()
	{
		return true;
	}

	/**
	 * エンジン情報追加処理。
	 */	
	class AddInfoAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * エンジン情報を追加する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			engineInfoComponent.setEngineInfo(null);
			if (PreferenceDialog.showDialog(PreferenceComponent.this, engineInfoComponent)) {
				EngineInfo info = engineInfoComponent.getEngineInfo();
				if (info != null) {
					PreferenceManager pm = PreferenceManager.getInstance();
					pm.addEngineInfo(info);
				}
			}
			engineInfoTable.updateUI();
		}
	}
	
	/**
	 * エンジン情報編集処理
	 */	
	class EditInfoAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * エンジン情報を編集する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			int selected = engineInfoTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("W9001"));
				return;
			}
			PreferenceManager pm = PreferenceManager.getInstance();
			EngineInfo info = pm.getEngineInfo(selected);
			engineInfoComponent.setEngineInfo(info);
			if (PreferenceDialog.showDialog(PreferenceComponent.this, engineInfoComponent)) {
				EngineInfo updateinfo = engineInfoComponent.getEngineInfo();
				if (updateinfo != null) {
					pm.updateEngineInfo(selected, updateinfo);
				}
			}
			engineInfoTable.updateUI();
		}
	}
	
	/**
	 * エンジン情報削除処理。
	 */	
	class DeleteInfoAction extends AbstractAction
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * エンジン情報を削除する。
		 * 
		 * @param e	イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			int selected = engineInfoTable.getSelectedRow();
			if (selected < 0) {
				DialogUtilities.showWarning(rm.getResource("W9001"));
				return;
			}
			PreferenceManager pm = PreferenceManager.getInstance();
			pm.removeEngineInfo(selected);

			engineInfoTable.clearSelection();
			engineInfoTable.updateUI();
		}
	}
	
	/**
	 * エンジン情報テーブル制御モデル
	 */
	class EngineInfoTableModel extends AbstractTableModel
	{
		/** シリアルバージョンID */
		private static final long serialVersionUID = 1L;

		/**
		 * エンジン情報テーブル制御モデルを生成する。
		 */
		EngineInfoTableModel()
		{
		}

		/**
		 * データソースオブジェクトが管理する列数を返す。
		 * JTableはこのメソッドを使って、初期化時に作成および表示する
		 * 必要のある列数を判断する。
		 *
		 * @return	モデルの列数
		 */
		public int getColumnCount()
		{
			return columnNames.length;
		}

		/**
		 * データソースオブジェクトが管理するレコード数を返す。
		 * JTableはこのメソッドを使って、作成および表示する必要のある行数を判断する。
		 *
		 * @return	モデルの行数
		 */
		public int getRowCount()
		{
			PreferenceManager pm = PreferenceManager.getInstance();
			return pm.getEngineInfoCount();
		}

		/**
		 * colにある列の名前を返す。
		 * このメソッドは、テーブルの列ヘッダ名を初期化するために使われる。
		 *
		 * @param	col		列のインデックス
		 * @return	列ヘッダ名
		 */
		public String getColumnName(int col)
		{
			return columnNames[col];
		}

		/**
		 * セルの値を返す。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	指定されたセルにある値
		 */
		public Object getValueAt(int row, int col)
		{
			PreferenceManager pm = PreferenceManager.getInstance();
			EngineInfo info = pm.getEngineInfo(row);
			if (info == null) return "";
			
			if (col == 0) {
				return info.getName();
			} else if (col == 1) {
				return info.getURL();
			} else if (col == 2) {
				return info.getScope();
			} else if (col == 3) {
				return info.getUser();
			}
			return "";
		}

		/**
		 * 列のClassを返す。
		 *
		 * @param	col		列
		 * @return	モデルのオブジェクト値の共通の上位クラス
		 */
		public Class<?> getColumnClass(int col)
		{
			return String.class;
		}

		/**
		 * rowとcolにあるセルが編集可能な場合はtrueを返す。
		 * そうでない場合、セルでsetValueAt()を呼び出しても、そのセルの値は変更されない。
		 *
		 * @param	row		値が参照される行
		 * @param	col		値が参照される列
		 * @return	セルが編集可能な場合はtrue
		 */
		public boolean isCellEditable(int row, int col)
		{
			return false;
		}
	}
}
