/* $Id: ProcessFlowPanel.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JToolBar;

import jp.co.argo21.nautica.commons.swing.DialogUtilities;
import jp.co.argo21.nautica.commons.swing.ExceptionViewer;
import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.IconManager;
import jp.co.argo21.nautica.tool.wfadm.WorkflowAdminManager;
import jp.co.argo21.nautica.tool.wfadm.common.BusyTask;
import jp.co.argo21.nautica.tool.wfadm.common.PropertyDialog;
import jp.co.argo21.nautica.workflow.definition.ActivityDefinition;
import jp.co.argo21.nautica.workflow.definition.PackageDefinition;
import jp.co.argo21.nautica.workflow.definition.ProcessDefinition;
import jp.co.argo21.nautica.workflow.wfmc.Activity;
import jp.co.argo21.nautica.workflow.wfmc.ActivityState;
import jp.co.argo21.nautica.workflow.wfmc.Attribute;
import jp.co.argo21.nautica.workflow.wfmc.Process;
import jp.co.argo21.nautica.workflow.wfmc.ProcessState;

/**
 * プロセスフロー表示用のパネルである。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class ProcessFlowPanel extends JPanel
{
	/** シリアルバージョンID */
	private static final long serialVersionUID = 1L;

	/** ツールバー */
	private JToolBar toolbar;
	/** エンジンビューフレーム */
	private EngineViewFrame frame;
	/** フロー表示エリア */
	private ProcessFlowView view;
	/** プロセス情報 */
	private Process process;
	/** パッケージ */
	private PackageDefinition packageDefinition;
	
	/**
	 * プロセスフロー表示用のパネルを生成する。
	 *
	 * @param isDoubleBuffered
	 */
	public ProcessFlowPanel(EngineViewFrame frame, Process proc, PackageDefinition def)
	{
		super(new BorderLayout(), true);
		
		this.frame = frame;
		this.process = proc;
		this.packageDefinition = def;
		
		createFlowView();
		createToolBar();
	}
	
	/**
	 * プロセスIDを返します。
	 *
	 * @return プロセスID
	 */
	String getProcessID()
	{
		return process.getID();
	}
	
	/**
	 * ツールバーを作成します。
	 */
	private void createToolBar()
	{
		ResourceManager rm = WorkflowAdminManager.getResourceManager();

		toolbar = new JToolBar(JToolBar.VERTICAL);
		toolbar.setFloatable(false);

		Icon icon0 = IconManager.getIcon("property");
		JButton propButton = new JButton();
		propButton.setIcon(icon0);
		propButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.0"));
		propButton.addActionListener(new ShowPropertyAction());

		Icon icon1 = IconManager.getIcon("varlist");
		JButton varlistButton = new JButton();
		varlistButton.setIcon(icon1);
		varlistButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.1"));
		varlistButton.addActionListener(new ShowAttributeListAction());

		Icon icon2 = IconManager.getIcon("reload");
		JButton reloadButton = new JButton();
		reloadButton.setIcon(icon2);
		reloadButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.2"));
		reloadButton.addActionListener(new ReloadAction());

		Icon icon3 = IconManager.getIcon("suspend");
		JButton suspendButton = new JButton();
		suspendButton.setIcon(icon3);
		suspendButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.3"));
		suspendButton.addActionListener(new SuspendActivityAction());

		Icon icon4 = IconManager.getIcon("resume");
		JButton resumeButton = new JButton();
		resumeButton.setIcon(icon4);
		resumeButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.4"));
		resumeButton.addActionListener(new ResumeActivityAction());

		Icon icon5 = IconManager.getIcon("complete");
		JButton completeButton = new JButton();
		completeButton.setIcon(icon5);
		completeButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.5"));
		completeButton.addActionListener(new CompleteActivityAction());

		Icon icon6 = IconManager.getIcon("abort");
		JButton abortButton = new JButton();
		abortButton.setIcon(icon6);
		abortButton.setToolTipText(rm.getResource("ProcessFlowPanel.toolbar.label.6"));
		abortButton.addActionListener(new AbortProcessAction());

		toolbar.add(propButton);
		toolbar.add(varlistButton);
		toolbar.add(reloadButton);
		toolbar.addSeparator();
		toolbar.add(suspendButton);
		toolbar.add(resumeButton);
		toolbar.addSeparator();
		toolbar.add(completeButton);
		toolbar.add(abortButton);

		add(toolbar, BorderLayout.WEST);
	}

	/**
	 * フロー表示エリアを作成します。
	 */
	private void createFlowView()
	{
		EngineViewController evc = frame.getEngineViewController();
		int index = getProcessDefinitionIndex();
		List<ProcessDefinition> procs = packageDefinition.getProcesses();
		ProcessDefinition def = procs.get(index);
		view = new ProcessFlowView(evc, def, process);
		JScrollPane sp = new JScrollPane(view);
		add(sp, BorderLayout.CENTER);
	}

	/**
	 * アクティビティ定義のプロパティ情報を作成して返す。
	 *
	 * @param rm リソースマネージャ
	 * @param act アクティビティ定義情報
	 * @return プロパティ情報
	 */
	@SuppressWarnings("unused")
	private Properties getProperties(ResourceManager rm, ActivityDefinition act)
	{
		Properties prop = new Properties();
		
		prop.put(rm.getResource("ActivityDefinition.key.0"),  "" + act.getAffiliateBID());
		prop.put(rm.getResource("ActivityDefinition.key.1"),  "" + act.getApplicationID());
		prop.put(rm.getResource("ActivityDefinition.key.2"),  "" + act.getBlockID());
		prop.put(rm.getResource("ActivityDefinition.key.3"),  "" + act.getDescription());
		prop.put(rm.getResource("ActivityDefinition.key.4"),  "" + act.getEngineName());
		prop.put(rm.getResource("ActivityDefinition.key.5"),  "" + act.getID());
		prop.put(rm.getResource("ActivityDefinition.key.6"),  "" + act.getName());
		prop.put(rm.getResource("ActivityDefinition.key.7"),  "" + act.getParentBID());
		prop.put(rm.getResource("ActivityDefinition.key.8"),  "" + act.getPerformerID());
		prop.put(rm.getResource("ActivityDefinition.key.9"),  "" + act.getSubprocessID());
		prop.put(rm.getResource("ActivityDefinition.key.10"), "" + act.getType());
		prop.put(rm.getResource("ActivityDefinition.key.11"), "" + act.getExecution());
		prop.put(rm.getResource("ActivityDefinition.key.12"), "" + act.getLimit());
		prop.put(rm.getResource("ActivityDefinition.key.13"), "" + act.getLogicalX());
		prop.put(rm.getResource("ActivityDefinition.key.14"), "" + act.getLogicalY());
		
		return prop;
	}

	/**
	 * アクティビティのプロパティ情報を作成して返す。
	 *
	 * @param rm リソースマネージャ
	 * @param act アクティビティ情報
	 * @return プロパティ情報
	 */
	private Properties getProperties(ResourceManager rm, Activity act)
	{
		Properties prop = new Properties();
		
		prop.put(rm.getResource("Activity.key.0"),  "" + act.getActivityDefinitionID());
		prop.put(rm.getResource("Activity.key.1"),  "" + act.getActivityID());
		prop.put(rm.getResource("Activity.key.2"),  "" + act.getActivityState());
		prop.put(rm.getResource("Activity.key.3"),  "" + act.getActorID());
		prop.put(rm.getResource("Activity.key.4"),  "" + act.getBlockID());
		prop.put(rm.getResource("Activity.key.5"),  "" + act.getName());
		prop.put(rm.getResource("Activity.key.6"),  "" + act.getProcessID());
		prop.put(rm.getResource("Activity.key.7"),  "" + act.getStarterID());
		prop.put(rm.getResource("Activity.key.8"),  "" + act.getDurationLimit());
		prop.put(rm.getResource("Activity.key.9"),  "" + act.getStartDate());
		prop.put(rm.getResource("Activity.key.10"), "" + act.getUpdateDate());
		
		return prop;
	}
	
	/**
	 * プロセスのインデクス番号を返す。
	 *
	 * @return インデクス番号
	 */
	private int getProcessDefinitionIndex()
	{
		String pdid = process.getProcDefinitionID();
		
		int begin = pdid.lastIndexOf('-') + 1;
		int end = pdid.length();
		String indexString = pdid.substring(begin, end);
		int index = Integer.parseInt(indexString);
		return index;
	}
	
	/**
	 * プロパティを表示するアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class ShowPropertyAction implements ActionListener
	{
		/**
		 * プロパティを表示する。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if (view == null) return;
			
			ResourceManager rm = WorkflowAdminManager.getResourceManager();

			ActivityDefinition actdef = view.getSelectedActivity();
			if (actdef == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}

			Map<String, Properties> props = new Hashtable<String, Properties>();
			String adid = actdef.getID();
			String adname = actdef.getName();
			List<Activity> acts = view.getActivitiesByDefinitionID(adid);
			if (acts.size() == 0) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}
			
			for (Activity act : acts) {
				String aname = act.getActivityID();
				Properties actprop = getProperties(rm, act);
				props.put(aname, actprop);
			}

			PropertyDialog.showDialog(ProcessFlowPanel.this, adname, props);
		}
	}
	
	/**
	 * プロセス変数を表示するアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class ShowAttributeListAction implements ActionListener
	{
		/**
		 * プロセス変数を表示する。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if (view == null) return;

			GetAttributeTask gvtask = new GetAttributeTask();
			WorkflowAdminManager.runBusyTask(gvtask);

			List<Attribute> list = gvtask.getAttributes();
			if (list == null || list.size() == 0) {
				ResourceManager rm = WorkflowAdminManager.getResourceManager();
				DialogUtilities.showInfo(rm.getResource("I9003"));
				return;
			}
			AttributeDialog dialog = new AttributeDialog();
			dialog.setAttributes(list);
			dialog.setVisible(true);
			int result = dialog.getResult();
			if (result == AttributeDialog.END_OPTION) {
				return;
			}
			List<Attribute> updatedList = dialog.getUpdatedAttributes();
			dialog.dispose();

			if (updatedList == null || updatedList.size() == 0) {
				ResourceManager rm = WorkflowAdminManager.getResourceManager();
				DialogUtilities.showInfo(rm.getResource("I9005"));
				return;
			}

			SetAttributeTask svtask = new SetAttributeTask(updatedList);
			WorkflowAdminManager.runBusyTask(svtask);
		}
	}
	
	/**
	 * フローの再表示を行うアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class ReloadAction implements ActionListener
	{
		/**
		 * フローの再表示を行う。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if (view == null) return;
			
			view.reload();
		}
	}
	
	/**
	 * アクティビティを完了状態にするアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class CompleteActivityAction implements ActionListener
	{
		/**
		 * アクティビティを完了状態にする。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowAdminManager.getResourceManager();

			ActivityDefinition actdef = view.getSelectedActivity();
			if (actdef == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}
			String adid = actdef.getID();
			Activity act = view.getActivityByDefinitionID(adid);
			
			if (act == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}

			int st = act.getActivityState();
			if (st == ActivityState.CLOSED_COMPLETED.toInt()
					|| st == ActivityState.CLOSED_TERMINATED.toInt()
					|| st == ActivityState.CLOSED_ABORTED.toInt()) {
				DialogUtilities.showError(rm.getResource("E9031"));
				return;
			}
			
			String pid = process.getID();
			String aid = act.getActivityID();
			BusyTask task = new ChangeActivityStateTask(pid, aid, ActivityState.CLOSED_COMPLETED);
			WorkflowAdminManager.runBusyTask(task);
			
			view.reload();
		}
	}
	
	/**
	 * アクティビティを保留状態にするアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class SuspendActivityAction implements ActionListener
	{
		/**
		 * アクティビティを保留状態にする。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			ResourceManager rm = WorkflowAdminManager.getResourceManager();

			ActivityDefinition actdef = view.getSelectedActivity();
			if (actdef == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}
			String adid = actdef.getID();
			Activity act = view.getActivityByDefinitionID(adid);
			
			if (act == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}

			int st = act.getActivityState();
			if (st != ActivityState.OPEN_RUNNING.toInt()) {
				DialogUtilities.showError(rm.getResource("E9028"));
				return;
			}
			
			String pid = process.getID();
			String aid = act.getActivityID();
			BusyTask task = new ChangeActivityStateTask(pid, aid, ActivityState.OPEN_NOT_RUNNING_SUSPENDED);
			WorkflowAdminManager.runBusyTask(task);
			
			view.reload();
		}
	}
	
	/**
	 * アクティビティを再開するアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class ResumeActivityAction implements ActionListener
	{
		/**
		 * アクティビティを再開する。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if (view == null) return;
			
			ResourceManager rm = WorkflowAdminManager.getResourceManager();

			ActivityDefinition actdef = view.getSelectedActivity();
			if (actdef == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}
			String adid = actdef.getID();
			Activity act = view.getActivityByDefinitionID(adid);
			
			if (act == null) {
				DialogUtilities.showError(rm.getResource("E9018"));
				return;
			}
			
			int st = act.getActivityState();
			if (st != ActivityState.OPEN_NOT_RUNNING_SUSPENDED.toInt()) {
				DialogUtilities.showError(rm.getResource("E9030"));
				return;
			}

			String pid = process.getID();
			String aid = act.getActivityID();
			BusyTask task = new ChangeActivityStateTask(pid, aid, ActivityState.OPEN_RUNNING);
			WorkflowAdminManager.runBusyTask(task);
			
			view.reload();
		}
	}
	
	/**
	 * プロセスを強制終了するアクション。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class AbortProcessAction
	implements ActionListener, BusyTask
	{
		/** 処理状態 */
		private int state = INIT;
		
		/**
		 * プロセスを強制終了する。
		 *
		 * @param e イベント
		 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
		 */
		public void actionPerformed(ActionEvent e)
		{
			if (view == null) return;
			if (process == null) return;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			
			int st = process.getProcessState();
			if (st == ProcessState.CLOSED_COMPLETED.toInt()
					|| st == ProcessState.CLOSED_TERMINATED.toInt()
					|| st == ProcessState.CLOSED_ABORTED.toInt()) {
				DialogUtilities.showError(rm.getResource("E9026"));
				return;
			}

			int ret = DialogUtilities.showYesNoConfirm(rm.getResource("I9004"));
			if (ret == JOptionPane.NO_OPTION) return;

			WorkflowAdminManager.runBusyTask(this);
			
			view.reload();
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}

		/**
		 * プロセスを強制終了する。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			String pid = process.getID();
			try {
				EngineViewController evc = frame.getEngineViewController();
				evc.changeProcessState(pid, ProcessState.CLOSED_ABORTED);
			} catch (Exception ex) {
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9027") + "(" + pid + ")");
			} finally {
				state = COMPLETE;
			}
		}
	}
	
	/**
	 * プロセス変数取得用タスク。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class GetAttributeTask implements BusyTask
	{
		/** 処理状態 */
		private int state = INIT;
		/** プロセス変数一覧 */
		private List<Attribute> attributes;
		
		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}

		/**
		 * プロセス変数を取得する。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			String pid = process.getID();
			try {
				EngineViewController evc = frame.getEngineViewController();
				attributes = evc.getProcessAttributes(pid, null);
			} catch (Exception ex) {
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9027"));
			} finally {
				state = COMPLETE;
			}
		}
		
		/**
		 * プロセス変数を返す。
		 *
		 * @return プロセス変数
		 */
		private List<Attribute> getAttributes()
		{
			return attributes;
		}
	}
	
	/**
	 * プロセス変数設定用タスク。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class SetAttributeTask implements BusyTask
	{
		/** 処理状態 */
		private int state = INIT;
		/** プロセス変数一覧 */
		private List<Attribute> attributes;
		
		/**
		 * プロセス変数設定用タスクを生成する。
		 *
		 * @param attributes プロセス変数一覧
		 */
		public SetAttributeTask(List<Attribute> attributes)
		{
			this.attributes = attributes;
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}

		/**
		 * プロセス変数を設定する。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			String pid = process.getID();
			try {
				EngineViewController evc = frame.getEngineViewController();
				evc.setProcessAttributes(pid, attributes);
			} catch (Exception ex) {
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9032"));
			} finally {
				state = COMPLETE;
			}
		}
	}
	
	/**
	 * アクティビティの状態を変更する処理。
	 *
	 * @author  nito(Argo 21, Corp.)
	 * @version $Revision: 1.2 $
	 * @since   Nautica Workflow 1.0
	 */
	private class ChangeActivityStateTask implements BusyTask
	{
		/** 処理状態 */
		private int state = INIT;
		
		/** 処理対象プロセスID */
		private String pid;
		/** 処理対象アクティビティID */
		private String aid;
		/** 処理対象アクティビティ状態 */
		private ActivityState astate;

		/**
		 * アクティビティの状態を変更する処理を生成する。
		 *
		 * @param pid プロセスID
		 * @param aid アクティビティID
		 * @param astate アクティビティ状態
		 */
		private ChangeActivityStateTask(String pid, String aid, ActivityState astate)
		{
			this.pid = pid;
			this.aid = aid;
			this.astate = astate;
		}

		/**
		 * 処理状態を返す。
		 *
		 * @return 処理状態
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#getState()
		 */
		public int getState()
		{
			return state;
		}

		/**
		 * アクティビティの状態を変更する。
		 * 
		 * @see jp.co.argo21.nautica.tool.wfadm.common.BusyTask#run()
		 */
		public void run()
		{
			state = EXEC;

			ResourceManager rm = WorkflowAdminManager.getResourceManager();
			try {
				if (aid == null) return;
				EngineViewController evc = frame.getEngineViewController();
				evc.changeActivityState(pid, aid, astate);
			} catch (Exception ex) {
				ExceptionViewer ev = WorkflowAdminManager.getExceptionViewer();
				ev.setThrowable(ex);
				DialogUtilities.showError(rm.getResource("E9029") + "(" + pid + "/" + aid + ")");
			} finally {
				aid = null;
				state = COMPLETE;
			}
		}
	}
}
