/* $Id: EngineNode.java,v 1.2 2007/12/10 09:34:42 nito Exp $
 *
 * Copyright (c)ARGO 21, Corporation. 2005, 2006.  All rights reserved.
 * 
 * This file is part of Nautica Workflow.
 * 
 *  Nautica Workflow is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 * 
 *  Nautica Workflow is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * 
 *  You should have received a copy of the GNU General Public License
 *  along with Nautica Workflow; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *  
 */
package jp.co.argo21.nautica.tool.wfadm.engineview;

import java.util.Enumeration;
import java.util.Properties;
import java.util.Vector;

import javax.swing.tree.TreeNode;

import jp.co.argo21.nautica.commons.util.ResourceManager;
import jp.co.argo21.nautica.tool.wfadm.pref.EngineInfo;
import jp.co.argo21.nautica.workflow.wfmc.Filter;
import jp.co.argo21.nautica.workflow.wfmc.ProcDef;

/**
 * エンジンツリーのトップノードである。
 * 子ノードとしてプロセス定義ノードを持つ。
 *
 * @author  nito(Argo 21, Corp.)
 * @version $Revision: 1.2 $
 * @since   Nautica Workflow 1.0
 */
public class EngineNode implements EngineTreeNode
{
	/** エンジンビューフレーム */
	private EngineViewFrame engineViewFrame;
	
	/** 子ノードの配列 */
	private Vector<ProcessDefinitionNode> children;
	
	/** 現在設定中のフィルター */
	private Filter currentFilter;
	
	/**
	 * エンジンツリーのトップノードを生成する。
	 * これはエンジンビューにひとつ存在する。
	 *
	 * @param frame エンジンビューフレーム
	 */
	public EngineNode(EngineViewFrame frame)
	{
		engineViewFrame = frame;
		children = new Vector<ProcessDefinitionNode>();
	}

	/**
	 * 子ノードをEnumerationで返す。
	 *
	 * @return 子ノード
	 * @see javax.swing.tree.TreeNode#children()
	 */
	public Enumeration<?> children()
	{
		return children.elements();
	}

	/**
	 * 子ノードを受け入れるかどうかを返す。
	 *
	 * @return 判定値
	 * @see javax.swing.tree.TreeNode#getAllowsChildren()
	 */
	public boolean getAllowsChildren()
	{
		return false;
	}

	/**
	 * インデクスで指定された子ノードを返す。
	 *
	 * @param childIndex インデクス
	 * @return 子ノード
	 * @see javax.swing.tree.TreeNode#getChildAt(int)
	 */
	public TreeNode getChildAt(int childIndex)
	{
		return (TreeNode)children.get(childIndex);
	}

	/**
	 * 子ノードの数を返す。
	 *
	 * @return 子ノード数
	 * @see javax.swing.tree.TreeNode#getChildCount()
	 */
	public int getChildCount()
	{
		return children.size();
	}

	/**
	 * 指定されたノードのインデクス値を返す。
	 *
	 * @param node ノード
	 * @return インデクス
	 * @see javax.swing.tree.TreeNode#getIndex(javax.swing.tree.TreeNode)
	 */
	public int getIndex(TreeNode node)
	{
		return children.indexOf(node);
	}
	
	/**
	 * 親ノードを返す。トップノードなので、常にnullを返す。
	 *
	 * @return 常にnull
	 * @see javax.swing.tree.TreeNode#getParent()
	 */
	public TreeNode getParent()
	{
		return null;
	}

	/**
	 * 末端ノードかどうかを返す。
	 *
	 * @return 末端ノードならtrue
	 * @see javax.swing.tree.TreeNode#isLeaf()
	 */
	public boolean isLeaf()
	{
		return false;
	}
	
	/**
	 * ノード名を返す。
	 *
	 * @return
	 * @see java.lang.Object#toString()
	 */
	public String toString()
	{
		EngineViewController evc = engineViewFrame.getEngineViewController();
		EngineInfo info = evc.getEngineInfo();
		
		return info.getName();
	}
	
	/**
	 * ノードの情報を更新する。
	 *
	 * @param filter
	 * @throws Exception
	 * @see jp.co.argo21.nautica.tool.wfadm.engineview.EngineTreeNode#reload(jp.co.argo21.nautica.workflow.wfmc.Filter)
	 */
	public void reload(Filter filter) throws Exception
	{
		if (filter != null) {
			currentFilter = filter;
		}
		EngineViewController evc = engineViewFrame.getEngineViewController();
		ProcDef[] defs = evc.getProcessDefinition(currentFilter);
		children.clear();
		for (int i = 0; i < defs.length; i++) {
			ProcessDefinitionNode child = new ProcessDefinitionNode(this, defs[i]);
			children.add(child);
		}
	}
	
	/**
	 * ノードのプロパティ情報を返す。
	 *
	 * @param rm リソースマネージャ
	 * @return プロパティ
	 * @see jp.co.argo21.nautica.tool.wfadm.engineview.EngineTreeNode#getProperties()
	 */
	public Properties getProperties(ResourceManager rm)
	{
		Properties prop = new Properties();

		EngineViewController evc = engineViewFrame.getEngineViewController();
		EngineInfo info = evc.getEngineInfo();
		
		prop.put(rm.getResource("EngineNode.key.0"), info.getName());
		prop.put(rm.getResource("EngineNode.key.1"), info.getURL());
		prop.put(rm.getResource("EngineNode.key.2"), info.getScope());
		prop.put(rm.getResource("EngineNode.key.3"), info.getUser());
		prop.put(rm.getResource("EngineNode.key.4"), info.getPassword());
		prop.put(rm.getResource("EngineNode.key.5"), info.getAdminKey());
		
		return prop;
	}

	/**
	 * エンジンビューフレームを返す。
	 *
	 * @return エンジンビューフレーム
	 */
	EngineViewFrame getFrame()
	{
		return engineViewFrame;
	}
}
