/*
 * fs.c
 *
 * Copyright 2002, Minoru Murashima. All rights reserved.
 * Distributed under the terms of the BSD License.
 *
 * ۥե륷ƥࡣ
 * ե륷ƥ
 */


#include<types.h>
#include<config.h>
#include<lib.h>
#include<mm.h>
#include<proc.h>
#include<lock.h>
#include<fs.h>
#include<test.h>
#include<errno.h>
#include<interrupt.h>
#include<time.h>
#include<net/net.h>
#include<share/utime.h>


/***************************************************************************************
 *
 * < ۥե륷ƥ९饹 >
 *
 *  ޥȻȥǥ쥯ȥѹե륪ץե¹Իɲä롣
 *  ȥ꡼ϡ"."+".."+̾ͽ¤٤롣
 *  ȥ꡼󥯤κǸϺǽΥȥ꡼Ϣ뤹롣
 *  ǥ쥯ȥϥޥȤ줿ΰʳϥǥХ֥åǤ϶
 *  ƥॳǸüȥ꡼ϡΥƥॳ˺ʤ褦
 *  󥿤䤷ΥƥॳκǸǥ󥿤򸺤餹
 *
 ***************************************************************************************/

enum{
	VFS_MAX_NAME_SIZE=255,		/* ۥե륷ƥκ̾ */
};


typedef struct{
	const char *path;
	VENTRY *vent;
}SEARCH_PARAM;


/* PUBLIC 롼ȥȥ꡼ */
static struct ROOTENT{
	VENTRY vent;
	char name[1];
}_rootEnt={
	{(VENTRY*)&_rootEnt,(VENTRY*)&_rootEnt,0,(VENTRY*)&_rootEnt,NULL,NULL,NULL,0,NULL,1,0,DIRECTORY,1},
	{'/'},
};
static VENTRY *rootEnt=(VENTRY*)&_rootEnt;

/* PUBLIC devǥ쥯ȥꥨȥ꡼ */
static struct DEVENT{
	VENTRY vent;
	char name[3];
}_devEnt={
	{(VENTRY*)&_devEnt,(VENTRY*)&_devEnt,0,(VENTRY*)&_rootEnt,NULL,NULL,NULL,0,NULL,1,0,DIRECTORY,3},
	{'d','e','v'},
};
static VENTRY *devEnt=(VENTRY*)&_devEnt;

/* PUBLIC ۥե륷ƥॹԥåȡ */
static int spinLock=0;


/*
 * PRIVATE
 * ȥ꡼󥯤饨ȥ꡼򸡺롣
 * Υȥ꡼ɥ쥹ݥ󥿡ˤϸĤäȥ꡼ޤϡ
 * ǸΥȥ꡼ɥ쥹Хåե롣
 * parameters : ѥɥ쥹,ǥ쥯ȥꥨȥ꡼ɥ쥹ݥ󥿡,entry buffer
 * return : next path or failed=NULL
 */
static const char *searchEntry(const char *path,VENTRY *entry,VENTRY **buf)
{
	int rest;
	VENTRY *last;


	if(cmpPathLength(".",path,1)==0)
	{
		*buf=entry;
		if(path[1]=='/')return &path[2];
		else return &path[1];
	}
	if(cmpPathLength("..",path,2)==0)
	{
		*buf=entry->parent;
		if(path[2]=='/')return &path[3];
		else return &path[2];
	}

	if((last=entry=entry->child)==NULL)
	{
		*buf=NULL;
		return NULL;
	}

	switch(rest=cmpPathLength(entry->name,path,entry->len))
	{
		case 1:
			*buf=NULL;
			return NULL;
		case -1:
			for(entry=entry->next;entry!=last;entry=entry->next)
				if((rest=cmpPathLength(entry->name,path,entry->len))!=-1)break;
	}

	*buf=entry;

	if(rest==0)
	{
		if(path[entry->len]=='/')return &path[entry->len+1];
		return &path[entry->len];
	}

	return NULL;
}


/*
 * PRIVATE
 * ȥ꡼롣
 * parameters : entry addres
 * return : 0 or error number
 */
static int delEntry(VENTRY *entry)
{
	VENTRY *parent;


	if(entry->refCount>0)return -EBUSY;
	if(entry->child!=NULL)return -EBUSY;

	parent=entry->parent;

	if(entry->next==entry)parent->child=NULL;
	else
	{
		if(parent->child==entry)parent->child=entry->next;
		entry->prev->next=entry->next;
		entry->next->prev=entry->prev;
	}

	kfree(entry->wait);
	kfree(entry);

	return 0;
}


/*
 * PRIVATE
 * ȥ꡼̤äƺ롣
 * parameters : ȥ꡼
 */
static inline void delUpEntry(VENTRY *entry)
{
	VENTRY *parent;

	for(;;)
	{
		parent=entry->parent;
		if(delEntry(entry)!=0)break;
		entry=parent;
	}
}


/*
 * PUBLIC
 * parameters : entry
 */
/* ӥե饰ΩƤ롣 */
static inline void setBusy(VENTRY *entry)
{
	entry->busy=1;
}

/* ӥե饰ݤ */
static inline void resetBusy(VENTRY *entry)
{
	entry->busy=0;
}

/* ӥե饰򸡺롣 */
static inline int isBusy(VENTRY *entry)
{
	return entry->busy;
}


/*
 * PUBLIC
 * ѥ̾Υȥ꡼򸡺롣
 * ǽΰˤϸƤӽФͤͿ졢ޥȤ줿ǥ쥯ȥ꤬Ф롣
 * ΰϥѥκǽΥȥ꡼롣
 * parameters : start search parmeters struct,last search parmeters struct
 * return : 0 or error number
 */
static int searchPath(SEARCH_PARAM *start,SEARCH_PARAM *end)
{
	const char *path,*pt;
	VENTRY *entry;
	VENTRY *ent_buf;


	entry=start->vent;
	for(path=start->path;*path!='\0';)
	{
		if(entry->type!=DIRECTORY)return -ENOTDIR;
		if((pt=searchEntry(path,entry,&ent_buf))==NULL)break;
		path=pt;
		entry=ent_buf;
		if(entry->mount!=NULL)
		{
			start->path=path;
			start->vent=entry->mount;
			entry=entry->mount;
		}
	}

	end->path=path;
	end->vent=entry;

	return 0;
}


/*
 * PUBLIC
 * ѥƬΡ̾Υȥ꡼󥯤ɲä롣
 * parameters : path pointer,parent entry,file type
 * return : new entry address
 */
static VENTRY *addEntry(const char *path,VENTRY *entry,int type)
{
	int len;
	VENTRY *next_ent,*new_ent;


	/* ȥ꡼򸡺 */
	if(searchEntry(path,entry,&next_ent)!=NULL)return NULL;

	/* ȥ꡼κ */
	len=pathLen(path);
	if((new_ent=(VENTRY*)kmalloc(sizeof(VENTRY)+len))==NULL)return NULL;
	memset(new_ent,0,sizeof(VENTRY));
	new_ent->fs=entry->fs;
	new_ent->din=entry->din;
	new_ent->type=type;
	new_ent->parent=entry;
	new_ent->len=len;
	memcpy(new_ent->name,path,len);

	/* 󥯤ɲä롣 */
	if(next_ent!=NULL)
	{
		new_ent->prev=next_ent->prev;
		new_ent->next=next_ent;
		next_ent->prev->next=new_ent;
		next_ent->prev=new_ent;
	}
	else if(entry->child!=NULL)
	{
		next_ent=entry->child;
		new_ent->prev=next_ent->prev;
		new_ent->next=next_ent;
		next_ent->prev->next=new_ent;
		next_ent->prev=new_ent;
		entry->child=new_ent;
	}
	else
	{
		new_ent->prev=new_ent;
		new_ent->next=new_ent;
		entry->child=new_ent;
	}

	return new_ent;
}


/*
 * PUBLIC
 * ȥ꡼λȥ󥿤1䤹
 * parameters : ۥȥ꡼
 */
static inline void linkVentry(VENTRY *vent)
{
	++vent->refCount;
}


/*
 * PUBLIC
 * ȥ꡼λȥ󥿤򸺤餷0ʤ̥Ρɥȥ꡼̤äƺ롣
 * parameters : ۥȥ꡼
 */
static inline int unlinkVentry(VENTRY *vent)
{
	if(--vent->refCount<=0)
	{
		delUpEntry(vent);
		return 0;
	}

	return vent->refCount;
}


/*
 * PUBLIC
 * ȥ꡼ѥɲä롣
 * ա pathκǽ餬NULLʸǤʤȡ
 * parameters : path,ۥȥ꡼
 * return : last entry or error number
 */
static inline int mkEntry(const char *path,VENTRY *vent)
{
	for(;;++path)
	{
		if((vent=addEntry(path,vent,DIRECTORY))==NULL)
		{
			unlinkVentry(vent);
			return -ENOMEM;
		}
		path+=vent->len;

		if(*path=='\0')break;
	}

	return (int)vent;
}


/*
 * PUBLIC
 * ȥ꡼˿ȥ꡼ޥȤ롣
 * parameters : ޥȥȥ꡼,filesystem,device inode,root directory block
 * return : 0 or error number
 */
static int mountEntry(VENTRY *entry,FS *fs,int din,uint block)
{
	VENTRY *mount_ent;


	/* Ǥ˥ޥȤƤ뤫 */
	if(entry->mount!=NULL)return -EBUSY;

	/* ޥȥȥ꡼κ */
	if((mount_ent=(VENTRY*)kmalloc(sizeof(VENTRY)+entry->len))==NULL)return -ENOMEM;
	memset(mount_ent,0,sizeof(VENTRY));
	mount_ent->next=mount_ent->prev=mount_ent;
	mount_ent->block=block;
	mount_ent->parent=entry->parent;
	mount_ent->fs=fs;
	mount_ent->din=din;
	mount_ent->type=DIRECTORY;
	mount_ent->len=entry->len;
	memcpy(mount_ent->name,entry->name,entry->len);

	linkVentry(mount_ent);

	/* ޥȥȥ꡼˥ޥȤ롣 */
	linkVentry(entry);
	entry->mount=mount_ent;

	return 0;
}


/*
 * PUBLIC
 * ޥȡ
 * parameters : ޥȥȥ꡼
 * return : 0 or error number
 */
static int umountEntry(VENTRY *entry)
{
	int rest;
	FS *fs;
	VENTRY *mount_ent;


	if((mount_ent=entry->mount)==NULL)return -ENOENT;
	if(mount_ent->refCount>1)return -EBUSY;
	if(mount_ent->child!=mount_ent->child->next->next)return -EBUSY;

	/* ޥȥȥ꡼κ */
	fs=mount_ent->fs;
	mount_ent->refCount=0;
	if((rest=delEntry(mount_ent))<0)return rest;
	if((rest=fs->umount(mount_ent->din))<0)return rest;
	--entry->refCount;

	/* ޥȥȥ꡼ʤ */
	unlinkVentry(entry);

	return 0;
}


/*
 * PUBLIC
 * ѥ롼ȤޤäƥХåե˥ԡ롣
 * parameters : buffer,buffer size,ۥȥ꡼
 * return : write size or error=-1
 */
static int copyCurrentPath(char *buf,size_t size,VENTRY *vent)
{
	int len;


	if(vent==rootEnt)
	{
		buf[0]='/';

		return 1;
	}
	else
	{
		if((len=copyCurrentPath(buf,size,vent->parent))==-1)return -1;
		if(len+vent->len+1>size)return -1;
		memcpy(&buf[len],vent->name,vent->len);
		buf[len+vent->len]='/';

		return len+vent->len+1;
	}
}


/************************************************************************************
 *
 * < ե륷ƥ饹 >
 *
 ************************************************************************************/

/* PRIVATE */
static FS fsInfoTop;


/*
 * PUBLIC
 * Search file system.
 * parameters : file system name
 * return : FS address or NULL
 */
static inline FS *search_fs(const char *name)
{
	FS *p;


	for(p=fsInfoTop.next;p!=NULL;p=p->next)
		switch(strcmp(name,p->name))
		{
			case 0:
				return p;
			case -1:
				return NULL;
		}

	return NULL;
}


/*
 * GLOBAL
 * Regist filesystem.
 * parameters : Filesystem struct
 * return : 0 or Error=-1
 */
int regist_fs(FS *info)
{
	FS *p;


	p=&fsInfoTop;
	for(;p->next!=NULL;p=p->next)
		switch(strcmp(info->name,p->next->name))
		{
			case 0:
				return -EEXIST;
			case -1:
				goto EXIT;
		}
EXIT:
	info->next=p->next;
	p->next=info;

	return 0;
}


/************************************************************************************
 *
 * ץΥե
 *
 ************************************************************************************/

/*
 * PRIVATE
 * Get empty fd number.
 * parameter : FILE_STRUCT,fd number
 * return : empty number
 */
static inline int getEmtpyFdNum(FILE_STRUCT *fstruct,int num)
{
	while(fstruct->fd[num]!=NULL)
		if(++num>=MAX_FILE_OPEN)num=0;
	if((fstruct->next_fd=num+1)>=MAX_FILE_OPEN)fstruct->next_fd=0;

	return num;
}


/*
 * PRIVATE
 * եǥץ鳺ե롣
 * parameters : F_DSC
 */
static void delFromFd(F_DSC *fd)
{
	char path[VFS_MAX_NAME_SIZE];
	int len;


	/* 롼ȤޤǤΥѥ롣 */
	enter_spinlock(&spinLock);
	{
		if((len=copyCurrentPath(path,VFS_MAX_NAME_SIZE,fd->vent))>=VFS_MAX_NAME_SIZE)
			return;
	}
	exit_spinlock(&spinLock);

	/*  */
	rootEnt->fs->delete(path,rootEnt->din,rootEnt->block,NORMAL_FILE);
}


/*
 * PUBLIC
 * Get current directory entry.
 * return current directory entry
 */
static inline VENTRY *getCurrentDir()
{
	return ((FILE_STRUCT*)get_current_task()->file_struct)->current_dir;
}


/*
 * PUBLIC
 * Set current directory.
 */
static inline void setCurrentDir(VENTRY *vent)
{
	FILE_STRUCT *fstruct;


	fstruct=get_current_task()->file_struct;
	unlinkVentry(fstruct->current_dir);
	fstruct->current_dir=vent;
}


/*
 * PUBLIC
 * ץեγǧ
 * return : 0 or error number
 */
static inline int checkOpenNum()
{
	FILE_STRUCT *fstruct;


	fstruct=get_current_task()->file_struct;
	if(fstruct->fd_count>=MAX_FILE_OPEN)return -EMFILE;
	else return 0;
}


/*
 * PUBLIC
 * Get file descriptor flag.
 * parameters : file descriptor number,flag
 * return : flag or error number
 */
static inline int getFdFlag(int num)
{
	FILE_STRUCT *fstruct;


	if((uint)num>=MAX_FILE_OPEN)return -EBADF;
	fstruct=get_current_task()->file_struct;
	if(fstruct->fd[num]==NULL)return -EBADF;

	return fstruct->flag[num];
}


/*
 * PUBLIC
 * Set file descriptor flag.
 * parameters : file descriptor number,flag
 * return : 0 or error number
 */
static inline int setFdFlag(int num,int flag)
{
	FILE_STRUCT *fstruct;


	if((uint)num>=MAX_FILE_OPEN)return -EBADF;
	fstruct=get_current_task()->file_struct;
	if(fstruct->fd[num]==NULL)return -EBADF;
	fstruct->flag[num]=flag;

	return 0;
}


/*
 * PUBLIC
 * Get file open mode flag.
 * parameters : file descriptor number
 * returm : open mode flag
 */
static inline int getOpenFlag(int num)
{
	FILE_STRUCT *fstruct;


	if((uint)num>=MAX_FILE_OPEN)return -EBADF;
	fstruct=get_current_task()->file_struct;
	if(fstruct->fd[num]==NULL)return -EBADF;

	return fstruct->fd[num]->accessMode;
}


/*
 * PUBLIC
 * Set file open mode flag.
 * parameters : file descriptor number
 * returm : open mode flag
 */
static inline int setOpenFlag(int num,int flag)
{
	FILE_STRUCT *fstruct;


	if((uint)num>=MAX_FILE_OPEN)return -EBADF;
	fstruct=get_current_task()->file_struct;
	if(fstruct->fd[num]==NULL)return -EBADF;

	fstruct->fd[num]->accessMode&=~(O_APPEND|O_NONBLOCK);
	fstruct->fd[num]->accessMode|=flag&(O_APPEND|O_NONBLOCK);

	return 0;
}


/*
 * PUBLIC
 * եǥץϿ롣
 * parameters : ventry,access mode,fd flag
 * return : file descriptor number or error number
 */
static int setFd(VENTRY *vent,int amode,int flag)
{
	int num;
	FILE_STRUCT *fstruct;
	F_DSC *fd;


	fstruct=get_current_task()->file_struct;
	if((fd=kmalloc(sizeof(F_DSC)))==NULL)return -ENOMEM;

	/* եǥץν */
	fd->form=FD_FORM_LOCALFILE;
	fd->accessMode=amode;
	fd->offset=0;
	fd->refCount=1;
	fd->vent=vent;
	num=getEmtpyFdNum(fstruct,fstruct->next_fd);
	++fstruct->fd_count;
	fstruct->fd[num]=fd;
	fstruct->flag[num]=flag;

	return num;
}


/*
 * PUBLIC
 * Copy file descriptor.
 * parameters : ԡ FD number,ԡǾ FD number
 * return : file descriptor number or error number
 */
static int copyFd(int num1,int num2)
{
	FILE_STRUCT *fstruct;


	if((uint)num2>=MAX_FILE_OPEN)return -EBADF;

	fstruct=get_current_task()->file_struct;
	num2=getEmtpyFdNum(fstruct,num2);
	fstruct->fd[num2]=fstruct->fd[num1];
	fstruct->flag[num2]=0;					/* Clear close-on-exec flag. */

	linkVentry(fstruct->fd[num2]->vent);

	return num2;
}


/*
 * PUBLIC
 * եǥץ롣
 * parameters : FILE_STRUCT,fd number
 */
static inline void deleteFd(FILE_STRUCT *fstruct,int num)
{
	fstruct->fd[num]=NULL;
	fstruct->next_fd=num;
	--fstruct->fd_count;
}


/*
 * PUBLIC
 * file descriptorλȥ󥿤򸺤餹
 * parameters : FILE_STRUCT,fd number
 * return : 0 or error number
 */
static int unlinkFd(FILE_STRUCT *fstruct,int num)
{
	int rest=0;
	int din;
	uint block;
	F_DSC *fd;
	VENTRY *vent;
	FS *fs;


	fd=fstruct->fd[num];

	fd->refCount-=1;

	if(fd->form==FD_FORM_NETSOKET)
	{
		if(fd->refCount<=0)
			closeSocket((SOCKET*)fd);
	}
	else
	{
		vent=fd->vent;
		if(fd->refCount<=0)
		{
			/* õե饰ǧ */
			if(fd->accessMode&O_DEL)delFromFd(fd);

			kfree(fd);
		}
		fs=vent->fs;
		din=vent->din;
		block=vent->block;
		if(unlinkVentry(vent)==0)rest=fs->close(din,block);
	}
	deleteFd(fstruct,num);

	return rest;
}


/*
 * GLOBAL
 * fork˥եطι¤Τƥץ饳ԡ롣
 * parameters : parent file structure
 * return : child file structure or error=NULL
 */
FILE_STRUCT *cpy_file_struct(FILE_STRUCT *fstruct)
{
	int i,cnt;
	FILE_STRUCT *cp_struct;


	if((cp_struct=kmalloc(sizeof(FILE_STRUCT)))==NULL)return NULL;
	memcpy(cp_struct,fstruct,sizeof(FILE_STRUCT));

	/* եǥץλȥ󥿤1䤹 */
	for(cnt=fstruct->fd_count,i=0;cnt>0;++i)
		if(cp_struct->fd[i]!=NULL)
		{
			++cp_struct->fd[i]->refCount;
			if(cp_struct->fd[i]->form==FD_FORM_LOCALFILE)
				linkVentry(cp_struct->fd[i]->vent);
			--cnt;
		}

	return cp_struct;
}


/*
 * GLOBAL
 * exit()Υեطι¤Τ롣
 * parameters : FILE_STRUCT address
 * return : 0 or error=-1
 */
void releaseFileStruct(FILE_STRUCT *fstruct)
{
	int i;


	/* Release all of file descriptors. */
	for(i=0;i<MAX_FILE_OPEN;++i)
	{
		if(fstruct->fd[i]!=NULL)
		{
			unlinkFd(fstruct,i);
			if(--fstruct->fd_count<=0)break;
		}
	}

	kfree(fstruct);
}


/*
 * GLOBAL
 * եǥץinode֤
 * parameters : file descriptor
 * return : inode block or error=-1
 */
int get_inode(int fd)
{
	F_DSC *fdsc;


	if(fd>MAX_FILE_OPEN)return -1;
	if((fdsc=((FILE_STRUCT*)(get_current_task()->file_struct))->fd[fd])==NULL)return -1;

	return fdsc->vent->block;
}


/*
 * GLOBAL
 * execΥե빽¤Τν
 */
void initExecFs()
{
	FILE_STRUCT *fstruct;
	int i,cnt;


	fstruct=get_current_task()->file_struct;

	/* Close file descriptor. */
	for(cnt=fstruct->fd_count,i=0;cnt>0;++i)
		if(fstruct->fd[i]!=NULL)
		{
			--cnt;
			if(fstruct->flag[i]&FD_CLOEXEC)
			{
				kfree(fstruct->fd[i]);
				fstruct->fd[i]=NULL;
				fstruct->flag[i]=0;
			}
		}
}


/*
 * GLOBAL
 * ɥץѥե빽¤Τν
 * parameters : process
 * return : 0 or error number
 */
int init_file_struct(PROC *proc)
{
	FILE_STRUCT *fstruct;


	if((fstruct=kmalloc(sizeof(FILE_STRUCT)))==NULL)return -ENOMEM;
	memset(fstruct,0,sizeof(FILE_STRUCT));
	fstruct->current_dir=rootEnt;
	proc->file_struct=fstruct;

	return 0;
}


/*
 * GLOBAL
 * Set file descriptar,for network socket.
 * parameters : file descriptor
 * return : file descriptar number or error number
 */
int setToFd(void *fd)
{
	int num;
	FILE_STRUCT *fstruct;


	fstruct=get_current_task()->file_struct;
	if(fstruct->fd_count>=MAX_FILE_OPEN)return -EMFILE;
	num=getEmtpyFdNum(fstruct,fstruct->next_fd);
	fstruct->fd[num]=fd;
	++fstruct->fd_count;

	return num;
}


/*
 * GLOBAL
 * File descriptor롣
 * parameters : file descriptor number
 * return : file descriptor or NULL
 */
F_DSC *getFd(int num)
{
	if((uint)num>=MAX_FILE_OPEN)return NULL;
	return ((FILE_STRUCT*)get_current_task()->file_struct)->fd[num];
}


/************************************************************************************
 *
 * System call interface
 *
 ************************************************************************************/


enum{
	/* եեޥåȡ */
	FS_FORM_LNK= 0xA000,	/* symbolic link */
	FS_FORM_REG= 0x8000,	/* regular file */
	FS_FORM_BLK= 0x6000,	/* block device */
	FS_FORM_DIR= 0x4000,	/* directory */
	FS_FORM_CHR= 0x2000,	/* character device */
	FS_FORM_FIFO=0x1000,	/* fifo */
};


/*
 * PUBLIC
 * pathȥåץǥ쥯ȥ롣
 * parameters : ѥɥ쥹ݥ
 * return : Ȳۥǥ쥯ȥꥨȥ꡼
 */
static inline VENTRY *getTopDir(const char **path)
{
	if(**path=='/')
	{
		*path+=1;
		return rootEnt;
	}
	else
		return getCurrentDir();
}


/*
 * GLOBAL
 * ¹ԥե륪ץ̾Υץδάؿ
 * parameters : path,open file struct
 * return : 0 or error number
 */
int exec_open(const char *path,OPEN_F *open_f)
{
	uint inode;
	int rest;
	SEARCH_PARAM start,end;
	VENTRY *vent;


	if(*path=='\0')return -ENOENT;

	/* եõơۥȥ꡼롣 */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if((*end.path=='\0')&&(end.vent->type==DIRECTORY))
	{
		rest=-EISDIR;
		goto END;
	}

	if(*end.path=='\0')
	{
		linkVentry(end.vent);
		vent=end.vent;
	}
	else
	{
		if((rest=start.vent->fs->open(start.path,start.vent->din,start.vent->block,0,&inode))<0)
			goto END;

		enter_spinlock(&spinLock);
		{
			rest=mkEntry(end.path,end.vent);
		}
		exit_spinlock(&spinLock);

		if(rest<0)
		{
			start.vent->fs->close(start.vent->din,start.vent->block);
			goto END;
		}

		vent=(VENTRY*)rest;
		vent->block=inode;
		vent->type=NORMAL_FILE;
	}

	open_f->vent=vent;
	rest=0;
END:
	resetBusy(end.vent);

	return rest;
}


/*
 * GLOBAL
 * ¹ԥեɤ߹ߡ̾Υ꡼ɤδάؿ
 * parameters : open file struct,buffer,read size
 * return : read byte or error=-1
 */
int exec_read(OPEN_F *open_f,void *buf,size_t size)
{
	VENTRY *vent;


	vent=open_f->vent;
	return vent->fs->read(vent->din,vent->block,buf,size,open_f->offset);
}


/*
 * PUBLIC
 * ޥȥǥХ
 * parameters : device name,filesystem number
 * return : device inode or error number
 */
static inline int mountDevice(const char *dev_path)
{
	enum{DEV_NAME_SIZE=5};		/* "/dev/"Υ */

	int rest;
	uint inode;


	/* ǥХץ */
	if(cmpPathNull("/dev",dev_path)==NULL)return -EINVAL;

	rest=devEnt->mount->fs->open(dev_path+DEV_NAME_SIZE,0,0,0,&inode);
	if(rest<0)return rest;
	return inode;
}

int sys_mount(const char *dev_path,const char *fs_name,const char *path)
{
	int mount_din;
	int rest;
	uint root_blk;
	FS *fs;
	SEARCH_PARAM start,end;
	VENTRY *vent;
	DIR dir;


	/* Search file system. */
	if((fs=search_fs(fs_name))==NULL)return -ENODEV;

	/* ǥХץ */
	if((rest=mountDevice(dev_path))<0)return rest;
	mount_din=rest;

	/* ޥȴؿƤӽФ */
	if((rest=fs->mount(mount_din))<0)goto ERR;
	root_blk=rest;

	/* ۥե륷ƥΥǥ쥯ȥ򸡺롣 */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			goto ERR;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path!='\0')
	{
		if((rest=start.vent->fs->opendir(start.path,start.vent->din,start.vent->block,&dir))<0)goto ERR2;
		if((rest=mkEntry(end.path,end.vent))<0)goto ERR2;
		vent=(VENTRY*)rest;
	}
	else vent=end.vent;

	/* ޥȤ벾ۥǥ쥯ȥ롣 */
	if((rest=mountEntry(vent,fs,mount_din,root_blk))<0)goto ERR2;

	resetBusy(end.vent);

	return 0;
ERR2:
	resetBusy(end.vent);
ERR:
	fs->umount(mount_din);

	return rest;
}


int sys_umount(const char *path)
{
	int rest;
	SEARCH_PARAM start,end;


	/* ǥ쥯ȥõ */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if((*end.path!='\0')||(end.vent->type!=DIRECTORY))
	{
		rest=-ENOENT;
		goto END;
	}

	/* ޥȡ */
	rest=umountEntry(end.vent);
END:
	resetBusy(end.vent);

	return rest;
}


int sys_lseek(int fd_num,int offset,int flag)
{
	int rest=0;
	FSTAT fstat;
	F_DSC *fd;
	VENTRY *vent;


	if((fd=getFd(fd_num))==NULL)return -EBADF;
	vent=fd->vent;
	vent->fs->stat(0,vent->block,&fstat);
	switch(flag)
	{
		case SEEK_CUR:
			rest=fd->offset;
		case SEEK_SET:
			if((rest+=offset)>fstat.st_size)return -EOVERFLOW;
			break;
		case SEEK_END:
			rest=fstat.st_size;
			if((rest-=offset)<0)return -EINVAL;
			break;
		default:
			return -EINVAL;
	}

	return fd->offset=rest;
}


int sys_creat(const char *path,int flag,int mode)
{
	int rest;
	int block;
	SEARCH_PARAM start,end;
	VENTRY *vent;


	if(*path=='\0')return -ENOENT;

	/* ץΥå */
	if((rest=checkOpenNum())<0)return rest;

	/* եõơۥȥ꡼롣 */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		rest=-EEXIST;
		goto END;
	}

	if(mode&O_FIFO)mode|=FS_FORM_FIFO;
	else mode|=FS_FORM_REG;
	if((rest=start.vent->fs->creat(start.path,start.vent->din,start.vent->block,mode))<0)goto END;
	block=rest;

	enter_spinlock(&spinLock);
	{
		rest=mkEntry(end.path,end.vent);
	}
	exit_spinlock(&spinLock);
	if(rest<0)
	{
		start.vent->fs->close(start.vent->din,start.vent->block);
		goto END;
	}

	vent=(VENTRY*)rest;
	vent->block=block;
	if(mode&O_FIFO)
	{
		if((vent->wait=kmalloc(sizeof(WAIT_QUEUE)))==NULL)goto END;
		memset(vent->wait,0,sizeof(WAIT_QUEUE));
		vent->type=FIFO_FILE;
	}
	else vent->type=NORMAL_FILE;

	/* եǥץν */
	rest=setFd(vent,flag,FD_CLOEXEC);
END:
	resetBusy(end.vent);

	return rest;
}


/*
 * parameters : path,flag,mode
 * return : file descriptor number or error=-1
 */
int sys_open(const char *path,int flag,int mode)
{
	uint inode;
	int rest;
	SEARCH_PARAM start,end;
	VENTRY *vent;


	if((flag&(O_CREAT|O_EXCL))==(O_CREAT|O_EXCL))return sys_creat(path,flag,mode);

	/* ץΥå */
	if((rest=checkOpenNum())<0)return rest;

	/* եõơۥȥ꡼롣 */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		if(end.vent->type!=DIRECTORY)
		{
			linkVentry(end.vent);
			vent=end.vent;
		}
		else
		{
			rest=-EISDIR;
			goto END;
		}
	}
	else
	{
		rest=start.vent->fs->open(start.path,start.vent->din,start.vent->block,flag&O_TRUNC,&inode);
		if((rest==-ENOENT)&&(flag&O_CREAT))
		{
			mode|=FS_FORM_REG;
			rest=start.vent->fs->creat(start.path,start.vent->din,start.vent->block,mode);
		}
		if(rest<0)goto END;

		enter_spinlock(&spinLock);
		{
			rest=mkEntry(end.path,end.vent);
		}
		exit_spinlock(&spinLock);
		if(rest<0)
		{
			start.vent->fs->close(start.vent->din,start.vent->block);
			goto END;
		}

		vent=(VENTRY*)rest;
		vent->block=inode;
		if(mode&O_FIFO)
		{
			if((vent->wait=kmalloc(sizeof(WAIT_QUEUE)))==NULL)goto END;
			memset(vent->wait,0,sizeof(WAIT_QUEUE));
			vent->type=FIFO_FILE;
		}
		else vent->type=NORMAL_FILE;
	}

	/* եǥץϿ롣 */
	if((rest=setFd(vent,mode,FD_CLOEXEC))<0)goto END;
	if(flag&O_APPEND)sys_lseek(rest,0,SEEK_END);
END:
	resetBusy(end.vent);

	return rest;
}


/*
 * parameters : file descriptor number
 * return : 0 or error number
 */
int sys_close(int fd_num)
{
	F_DSC *fd;
	FILE_STRUCT *fstruct;


	if((fd=getFd(fd_num))==NULL)return -EBADF;

	fstruct=(FILE_STRUCT*)get_current_task()->file_struct;
	return unlinkFd(fstruct,fd_num);
}


/*
 * parameters : file descriptor number,baffer,read bytes
 * return : read bytes or error=-1
 */
int sys_read(int fd_num,void *buf,size_t size)
{
	F_DSC *fd;
	VENTRY *vent;
	int rest;


	if((fd=getFd(fd_num))==NULL)return -EBADF;
	if((fd->accessMode&(O_RDONLY|O_RDWR))==0)return -EBADF;		/* ⡼ɤΥå */

	/* readؿƤӽФ */
	vent=fd->vent;
	if(vent->type==FIFO_FILE)
	{
		waitSemaphore(vent->wait);
		{
			rest=vent->fs->read(vent->din,vent->block,buf,size,0);
		}
		wakeSemaphore(vent->wait);
	}
	else
		if((rest=vent->fs->read(vent->din,vent->block,buf,size,fd->offset))>0)
			fd->offset+=rest;

	return rest;
}


int sys_write(int fd_num,void *buf,size_t size)
{
	F_DSC *fd;
	VENTRY *vent;
	int rest;


	if((fd=getFd(fd_num))==NULL)return -EBADF;
	if((fd->accessMode&(O_WRONLY|O_RDWR))==0)return -EBADF;		/* ⡼ɤΥå */

	/* writeؿƤӽФ */
	vent=fd->vent;
	if(vent->type==FIFO_FILE)
	{
		waitSemaphore(vent->wait);
		{
			rest=vent->fs->write(vent->din,vent->block,buf,size,0);
		}
		wakeSemaphore(vent->wait);
	}
	else
	{
		rest=vent->fs->write(vent->din,vent->block,buf,size,fd->offset);
		if(rest==-1)return -1;
		fd->offset+=rest;
	}

	return rest;
}


int sys_mkdir(const char *path,int mode)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		rest=-EEXIST;
		goto END;
	}

	rest=start.vent->fs->mkdir(start.path,start.vent->din,start.vent->block,mode);
END:
	resetBusy(end.vent);

	return rest;
}


int sys_unlink(const char *path)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path!='\0')
		rest=start.vent->fs->delete(start.path,start.vent->din,start.vent->block,NORMAL_FILE);
	else rest=-EBUSY;

	resetBusy(end.vent);

	return rest;
}


int sys_rmdir(const char *path)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path!='\0')
		rest=start.vent->fs->delete(start.path,start.vent->din,start.vent->block,DIRECTORY);
	else rest=-EBUSY;

	resetBusy(end.vent);

	return rest;
}


int sys_opendir(const char *path,DIR *dir)
{
	int rest;
	SEARCH_PARAM start,end;


	if(*path=='\0')return -ENOENT;

	/* եõ */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(end.vent->type==DIRECTORY)
	{
		if((rest=start.vent->fs->opendir(start.path,start.vent->din,start.vent->block,dir))>=0)
		{
			dir->dinode=start.vent->din;
			dir->inode=rest;
			dir->fs=start.vent->fs;
			rest=0;
		}
	}
	else rest=-ENOENT;

	resetBusy(end.vent);

	return rest;
}


int sys_closedir(DIR *dir)
{
	return ((FS*)dir->fs)->close(dir->dinode,dir->inode);
}


int sys_readdir(DIR *dir,char *name)
{
	return ((FS*)dir->fs)->readdir(dir->dinode,dir->inode,dir,name);
}


/*
 * old pathnew pathƱ졢ͭå롣
 */
int sys_rename(const char *old_path,const char *new_path)
{
	int rest;
	SEARCH_PARAM old_start,old_end;
	SEARCH_PARAM new_start,new_end;


	if(strcmp(old_path,new_path)==0)return 0;

	/* եõ */
	old_start.vent=getTopDir(&old_path);
	old_start.path=old_path;
	new_start.vent=getTopDir(&new_path);
	new_start.path=new_path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if(((rest=searchPath(&old_start,&old_end))<0)||((rest=searchPath(&new_start,&new_end))<0))
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if((isBusy(old_end.vent)==0)&&(isBusy(new_end.vent)==0))
		{
			setBusy(old_end.vent);
			setBusy(new_end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(old_start.vent->din==new_start.vent->din)
	{
		if((*old_end.path!='\0')&&(*new_end.path!='\0'))
			rest=old_start.vent->fs->
				rename(old_start.vent->din,old_start.vent->block,old_start.path,new_start.vent->block,new_start.path);
		else rest=-EBUSY;
	}
	else rest=-EXDEV;

	resetBusy(old_end.vent);
	if(old_end.vent!=new_end.vent)resetBusy(new_end.vent);

	return rest;
}


int sys_chdir(const char *path)
{
	uint block;
	int rest=0;
	SEARCH_PARAM start,end;
	VENTRY *vent;
	DIR dir;


	if(*path=='\0')return -ENOENT;

	/* ǥ쥯ȥõ */
	start.vent=getTopDir(&path);
    start.path=path;
    for(;;)
    {
        enter_spinlock(&spinLock);

        if((rest=searchPath(&start,&end))<0)
        {
            exit_spinlock(&spinLock);
            return rest;
        }
        if(isBusy(end.vent)==0)
        {
            setBusy(end.vent);
            exit_spinlock(&spinLock);
            break;
        }
        exit_spinlock(&spinLock);
        wait_task();
    }

	if(end.vent->type==DIRECTORY)
	{
		if(*end.path=='\0')
		{
			linkVentry(end.vent);
			vent=end.vent;
		}
		else
		{
			if((rest=start.vent->fs->opendir(start.path,start.vent->din,start.vent->block,&dir))>0)
			{
				block=rest;
				enter_spinlock(&spinLock);
				{
					rest=mkEntry(end.path,end.vent);
				}
				exit_spinlock(&spinLock);

				if(rest>0)
				{
					vent=(VENTRY*)rest;
					vent->block=block;
					vent->type=DIRECTORY;
				}
				else goto ERR;
			}
			else goto ERR;
		}
	}
	else
	{
		rest=-ENOTDIR;
		goto ERR;
	}

	resetBusy(end.vent);

	/* ե빽¤Τ˿ȥǥ쥯ȥϿ */
	setCurrentDir(vent);

	return 0;
ERR:
	resetBusy(end.vent);

	return rest;
}


int sys_getcwd(char *buf,size_t size)
{
	int len;
	VENTRY *vent;


	if(checkMem(buf)==-1)return -EFAULT;
	if(size==0)return -EINVAL;

	vent=getCurrentDir();
	if((len=copyCurrentPath(buf,size,vent))==-1)return -ERANGE;
	if(len==1)buf[1]='\0';
	else buf[len-1]='\0';

	return 0;
}


/*
 * pathNULLʤfile descriptorõ
 * parameters : file descriptor,path,stat buffer
 */
int sys_stat(int fd_num,const char *path,struct stat *buf)
{
	uint inode;
	int rest;
	SEARCH_PARAM start,end;
	F_DSC *fd;
	DIR dir;


	if(checkMem(buf)==-1)return -EFAULT;

	if(path==NULL)
	{
		if((fd=getFd(fd_num))==NULL)return -EBADF;
		buf->st_crtpos=fd->offset;
		return fd->vent->fs->stat(0,fd->vent->block,buf);
	}
	else
	{
		buf->st_crtpos=0;

		start.vent=getTopDir(&path);
		start.path=path;
		if((rest=searchPath(&start,&end))<0)return rest;
		if((*end.path=='\0')&&(end.vent->block!=0))
			return end.vent->fs->stat(0,end.vent->block,buf);
		else
		{
			if((rest=start.vent->fs->open(start.path,start.vent->din,start.vent->block,0,&inode))==-EISDIR)
			{
				rest=start.vent->fs->opendir(start.path,start.vent->din,start.vent->block,&dir);
				inode=rest;
			}
			if(rest<0)return rest;
			rest=start.vent->fs->stat(0,inode,buf);
			start.vent->fs->close(start.vent->din,inode);

			return rest;
		}
	}
}


/*
 * PRIVATE
 * ۥե륷ƥ򸡺ƥޥȤƤե륷ƥ򥢥ޥȤ롣
 * parameters : vertial entry
 * return : process or Failed=NULL
 */
static void umountAllFs(VENTRY *ventry)
{
	VENTRY *vent=ventry;


	do
	{
		if(vent->child!=NULL)umountAllFs(vent->child);

		/* umouont file system. */
		if(vent->mount!=NULL)vent->mount->fs->umount(vent->mount->din);
	}while((vent=vent->next)!=ventry);
}


/*
 * 롼ȥե륷ƥ򥢥ޥȤ롣
 */
int sys_umount_root()
{
	umountAllFs(rootEnt);

	return 0;
}


int sys_fcntl(int fd_num,int cmd,uint arg)
{
	switch(cmd)
	{
		case F_DUPFD:
		{
			int rest;

			if((rest=checkOpenNum())<0)return rest;		/* ץΥå */
			if(getFd(fd_num)==NULL)return -EBADF;		/* File descriptorΥå */
			return copyFd(fd_num,arg);					/* Copy File descriptor. */
		}
		case F_GETFD:
			return getFdFlag(fd_num);
		case F_SETFD:
			return setFdFlag(fd_num,arg);
		case F_GETFL:
			return getOpenFlag(fd_num);
		case F_SETFL:
			return setOpenFlag(fd_num,arg);
		case F_GETLK:
		case F_SETLK:
		case F_SETLKW:
		case F_GETOWN:
		case F_SETOWN:
		default:
			return -ENOTSUP;
	}
}


int sys_chattr(const char *path,int cmd,uint attr)
{
	uint inode=0;
	int rest;
	int open=0;
	SEARCH_PARAM start,end;
	FS *fs;
	PROC *proc;
	struct stat stat;


	/* Open inode. */
	start.vent=getTopDir(&path);
	start.path=path;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
       	{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}
	if((*end.path=='\0')&&(end.vent->block!=0))
	{
		fs=end.vent->fs;
		inode=end.vent->block;
	}
	else
	{
		fs=start.vent->fs;
		if((rest=fs->open(start.path,start.vent->din,start.vent->block,0,&inode))<0)goto ERR;
		open=1;
	}

	/* Change attribute. */
	proc=get_current_task();
	if((rest=fs->stat(0,inode,&stat))<0)goto ERR;
	switch(cmd)
	{
		case CHATTR_MODE:
			if((proc->uid==0)||(proc->uid==stat.st_uid))
				rest=fs->chattr(0,inode,attr,stat.st_uid,stat.st_gid,stat.st_atime,stat.st_mtime);
			else rest=-EPERM;
			break;
		case CHATTR_UID:
			if(proc->uid==0)
				rest=fs->chattr(0,inode,stat.st_mode,attr,stat.st_gid,stat.st_atime,stat.st_mtime);
			else rest=-EPERM;
			break;
		case CHATTR_GID:
			if(((proc->uid==0)||(proc->uid==stat.st_uid))&&(attr==proc->gid))
				rest=fs->chattr(0,inode,stat.st_mode,stat.st_gid,attr,stat.st_atime,stat.st_mtime);
			else rest=-EPERM;
			break;
		case CHATTR_UTIME:
			if((proc->uid==0)||(proc->uid==stat.st_uid))
			{
				struct utimbuf *ut=(struct utimbuf*)attr;
				uint time;

				if(ut==NULL)
				{
					time=sys_time(NULL);
					rest=fs->chattr(0,inode,stat.st_mode,stat.st_gid,attr,time,time);
				}
				else
					rest=fs->chattr(0,inode,stat.st_mode,stat.st_gid,attr,ut->actime,ut->modtime);
			}
			else rest=-EPERM;
			break;
	}
ERR:
	if(open)fs->close(start.vent->din,inode);
	resetBusy(end.vent);

	return rest;
}


/* ɲðꡣ */
int sys_ioctl(int fd_num,int prm_num,int cmd)
{
	F_DSC *fd;
	VENTRY *vent;
	uint *arg=(uint*)&cmd+1;


	if((fd=getFd(fd_num))==NULL)return -EBADF;
	vent=fd->vent;
	switch(prm_num)
	{
		case 0:
			return vent->fs->ioctl(vent->din,prm_num,cmd);
		case 1:
			return vent->fs->ioctl(vent->din,prm_num,cmd,*arg);
		case 2:
			return vent->fs->ioctl(vent->din,prm_num,cmd,*arg,*(arg+1));
		default:
			return 0;
	}
}


int sys_link(const char *oldpath, const char *newpath)
{
	uint oldinode;
	int din;
	int rest;
	SEARCH_PARAM start,end;


	/* oldpathΥե򳫤 */
	start.vent=getTopDir(&oldpath);
	start.path=oldpath;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		if(end.vent->type==DIRECTORY)
		{
			rest=-EISDIR;
			goto END;
		}
		din=end.vent->din;
		oldinode=end.vent->block;
	}
	else
	{
		din=start.vent->din;
		if((rest=start.vent->fs->open(start.path,din,start.vent->block,0,&oldinode))<0)
			goto END;
	}
	resetBusy(end.vent);

	/* newpathΥե򸡺롣 */
	start.vent=getTopDir(&newpath);
	start.path=newpath;
	for(;;)
	{
		enter_spinlock(&spinLock);

		if((rest=searchPath(&start,&end))<0)
		{
			exit_spinlock(&spinLock);
			return rest;
		}
		if(isBusy(end.vent)==0)
		{
			setBusy(end.vent);
			exit_spinlock(&spinLock);
			break;
		}
		exit_spinlock(&spinLock);
		wait_task();
	}

	if(*end.path=='\0')
	{
		rest=-EEXIST;
		goto END;
	}
	else
	{
		if(start.vent->din!=din)rest=-EXDEV;
		else
			rest=start.vent->fs->mklink(din,oldinode,start.vent->block,start.path);
	}

END:
	resetBusy(end.vent);

	return rest;
}

/************************************* ̤ ******************************************/
int sys_lock()
{
	return 0;
}


/************************************************************************************
 *
 * 
 *
 ************************************************************************************/

/*
 * GLOBAL
 * Init filesystem
 */
void init_fs()
{
	/* Device filesystemȥ꡼ꡣ */
	rootEnt->child=devEnt;
}


/*
 * GLOBAL
 * Mount device filesystem.
 * parameters : filesystem
 * return : 0 or error number
 */
int mount_dev_fs(FS *fs)
{
	return mountEntry(devEnt,fs,0,0);
}


/*
 * GLOBAL
 * mount filesystem
 * parameters : device name,filesystem name
 * return : 0 or error number
 */
int mountRoot(const char *dev_path,const char *fs_name)
{
	int mount_din;
	uint root_blk;
	FS *fs;


	/* Search file system number */
	if((fs=search_fs(fs_name))==NULL)return -ENODEV;

	/* ǥХץ */
	if((mount_din=mountDevice(dev_path))<0)return mount_din;

	/* ޥȴؿƤӽФ */
	if((root_blk=fs->mount(mount_din))==-1)goto ERR;

	/* 롼Ȳۥǥ쥯ȥꡣ */
	rootEnt->din=mount_din;
	rootEnt->fs=fs;
	rootEnt->block=root_blk;

	return 0;
ERR:
	fs->close(0,mount_din);

	return -1;
}


/*
 * GLOBAL
 * ޥȥڡХååץǥХ
 * parameters : device name
 * return : device inode or error number
 */
int mountPageBackupDev(const char *dev_path)
{
	return mountDevice(dev_path);
}


/*
 * GLOBAL
 * Init process0 file struct.
 * return 0 or error number
 */
int initProc0Fs()
{
	int rest;


	/*
	 * ɸϤ򳫤
	 */
	if((rest=sys_open("/dev/console",0,O_RDONLY))<0)return rest;	/* ɸϡ */
	if((rest=setFdFlag(rest,0))<0)return rest;						/* No exec fd close. */
	if((rest=sys_open("/dev/console",0,O_WRONLY))<0)return rest;	/* ɸϡ */
	if((rest=setFdFlag(rest,0))<0)return rest;
	if((rest=sys_open("/dev/console",0,O_WRONLY))<0)return rest;	/* ɸ२顼 */
	if((rest=setFdFlag(rest,0))<0)return rest;

	return 0;
}

/*************************************************************************************************/
void test_fs()
{
	printk("block=%x\n",((VENTRY*)0x16cf80)->block);
}
/**************************************************************************************************/
