/*
 * Copyright (c) 2003, Morpho Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 3. Neither the name of the Morpho Project nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package jp.morpho.webapp.resource;

import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.URL;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.ServletContext;

import jp.morpho.config.Configuration;
import jp.morpho.util.ConversionList;
import jp.morpho.util.PrimitiveHolderList;
import jp.morpho.webapp.Plugin;
import jp.morpho.webapp.PluginContext;
import jp.morpho.webapp.PluginException;
import jp.morpho.webapp.PluginKeys;

/**
 * <P>
 * \[X[ŨP[ɈˑȂ@ō\z邽߂̂̂łB
 * Gh[Upɕ\郁bZ[ẂA̕@ō\zĂB
 * </P>
 * @author Kumiko Hiroi
 */
public class MessageResourceManager
	implements Plugin
{
	/** vOC ReLXg */
	private PluginContext pluginContext = null;

	/** ݒt@CpX̃Xg */
	private PrimitiveHolderList configPathList = new ConversionList();

	/** \[XohtB[h擾ƂɎgpL[̊ꖼ */
	private static final String fieldNameBaseKey = "fieldName.";

	/** bZ[W \[X Gg̃Xg */
	protected ArrayList resourceEntries = new ArrayList();

	/**
	 * <P>
	 * MessageResourceManager 쐬܂B
	 * </P>
	 */
	public MessageResourceManager()
	{
		super();
	}

	/**
	 * <P>
	 * ׂĂ̐ݒt@CpX폜܂B
	 * </P>
	 */
	public synchronized void removeAllConfigurationFilePaths()
	{
		configPathList.clear();
	}

	/**
	 * <P>
	 * ݒt@CpXǉ܂B
	 * ݒt@C̓t[[N pluginActivate() \bhĂяoꂽƂɓǂݍ܂܂B
	 * </P>
	 * @param path ݒt@CpX
	 */
	public synchronized void addConfigurationFilePath(String path)
	{
		configPathList.add(path);
	}

	/**
	 * <P>
	 * ݒt@CpXK؂ȏŊi[ĂzԂ܂B
	 * </P>
	 * @return ݒt@CpXK؂ȏŊi[Ăz
	 */
	public synchronized String[] getConfigurationFilePaths()
	{
		String[] paths = new String[configPathList.size()];
		return (String[])configPathList.toArray(paths);
	}

	/**
	 * <P>
	 * bZ[W \[X }l[WɎw̃Ggǉ܂B
	 * </P>
	 * @param resourceEntry bZ[W \[X }l[WɒǉGg
	 */
	public synchronized void addMessageResourceEntry(MessageResourceEntry resourceEntry)
	{
		resourceEntries.add(resourceEntry);
	}

	/**
	 * <P>
	 * ̃bZ[W \[X }l[Wɐݒ肳Ă邷ׂẴGgAw肳ꂽL[̃\[X擾ăbZ[W\z܂B
	 * </P>
	 * @param key 擾郊\[X̃L[
	 * @param objs tH[}bg邩܂͒uΏۂ̃IuWFNgȂz
	 * @param locale \[XohKvȃP[
	 * @return tH[}bgꂽbZ[W
	 */
	public synchronized String getMessage(String key, Object[] objs, Locale locale)
	{
		return getMessage(key, objs, locale, (String)null);
	}

	/**
	 * <P>
	 * w肳ꂽÕGgAw肳ꂽL[̃\[X擾ăbZ[W\z܂B
	 * </P>
	 * @param key 擾郊\[X̃L[
	 * @param objs tH[}bg邩܂͒uΏۂ̃IuWFNgȂz
	 * @param locale \[XohKvȃP[
	 * @param resourceName \[X擾Gg̖O
	 * @return tH[}bgꂽbZ[W
	 */
	public synchronized String getMessage(String key, Object[] objs, Locale locale, String resourceName)
	{
		if (resourceName != null)
		{
			for (int i = 0; i < resourceEntries.size(); i++)
			{
				MessageResourceEntry resourceEntry = (MessageResourceEntry)resourceEntries.get(i);
				if (resourceEntry.getResourceName().equals(resourceName))
				{
					return getMessage(key, objs, locale, resourceEntry);
				}
			}
		}
		else
		{
			for (int i = 0; i < resourceEntries.size(); i++)
			{
				MessageResourceEntry resourceEntry = (MessageResourceEntry)resourceEntries.get(i);
				try
				{
					return getMessage(key, objs, locale, resourceEntry);
				}
				catch (Exception e)
				{
				}
			}
		}

		throw new MessageResourceException("Can't find resource for key=" + key);
	}

	/**
	 * <P>
	 * w肳ꂽGgAw肳ꂽL[̃\[X擾ăbZ[W\z܂B
	 * </P>
	 * @param key 擾郊\[X̃L[
	 * @param objs tH[}bg邩܂͒uΏۂ̃IuWFNgȂz
	 * @param locale \[XohKvȃP[
	 * @param entry \[X擾Gg
	 * @return tH[}bgꂽbZ[W
	 */
	protected String getMessage(String key, Object[] objs, Locale locale, MessageResourceEntry entry)
	{
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		if (cl == null)
		{
			cl = getClass().getClassLoader();
		}

		String baseName = entry.getBaseName();
		String className = entry.getClassName();
		Class cls = null;
		try
		{
			cls = cl.loadClass(className);
		}
		catch (Exception e)
		{
			throw new MessageResourceException(e.getMessage());
		}

		Method method = null;
		try
		{
			method =
				cls.getMethod(
					"getBundle",
					new Class[]
						{ String.class, Locale.class, ClassLoader.class });
		}
		catch (Exception e)
		{
			throw new MessageResourceException(e.getMessage());
		}

		ResourceBundle rb = null;
		try
		{
			rb =
				(ResourceBundle) method.invoke(
					cls,
					new Object[] { baseName, locale, cl });
		}
		catch (Exception e)
		{
			throw new MessageResourceException(e.getMessage());
		}

		try
		{
			String msg = rb.getString(key);
			if (objs == null)
			{
				return msg;
			}

			try
			{
				MessageFormat mf = new MessageFormat(msg);
				return mf.format(objs);
			}
			catch (Exception e)
			{
				throw new MessageResourceException(e.getMessage());
			}
		}
		catch (MessageResourceException e)
		{
			throw e;
		}
		catch (Exception e)
		{
			throw new MessageResourceException(e.getMessage());
		}
	}

	/**
	 * <P>
	 * ̃bZ[W \[X }l[Wɐݒ肳Ă邷ׂẴGgAw肳ꂽL[̃\[X擾܂B
	 * ͎ƓłB
	 * </P>
	 * <P>
	 * getMessage("fieldName." + key, null, locale);
	 * </P>
	 * @param key 擾郊\[X̃L[
	 * @param locale \[XohKvȃP[
	 * @return tB[h
	 */
	public synchronized String getFieldName(String key, Locale locale)
	{
		return getMessage(fieldNameBaseKey + key, null, locale);
	}

	/**
	 * <P>
	 * vOCʂ邽߂̃L[Ԃ܂B
	 * vOC̃L[́ApbP[WƓKɏ]Kv܂B
	 * ̎dlł́Ajp.morpho.* Ɉv閼O\񂵂Ă܂B
	 * </P>
	 * @return vOCʂ邽߂̃L[
	 * @see jp.morpho.webapp.Plugin#getPluginKey()
	 */
	public final String getPluginKey()
	{
		return PluginKeys.messageResourceManager;
	}

	/**
	 * <P>
	 * vOCReLXgݒ肵܂B
	 * t[[Nł́AvOCCX^Xǉꂽɂ̃\bhĂяo܂B
	 * </P>
	 * @param fc ̃CX^Xǉ FrameworkContext IuWFNg
	 * @exception PluginException ̃\bhɂăX[A炩̃G[ŏQƂ
	 */
	public void setPluginContext(PluginContext fc)
		throws PluginException
	{
		pluginContext = fc;
	}

	/**
	 * <P>
	 * ݒt@Cǂݍ݁AbZ[W \[X }l[W܂B
	 * </P>
	 * @exception PluginException ̃\bhɂăX[A炩̃G[ŏQƂ
	 */
	public void pluginActivate()
		throws PluginException
	{
		ServletContext sc = pluginContext.getFrameworkContext().getServletContext();
		String[] paths = getConfigurationFilePaths();
		for (int i = 0; i < paths.length; i++)
		{
			InputStream in = null;
			try
			{
				URL url = sc.getResource(paths[i]);
				in = url.openStream();
				Configuration config = new Configuration();
				config.configure(this, in);
			}
			catch (Exception e)
			{
				Logger log = Logger.getLogger("jp.morpho");
				log.log(Level.WARNING, e.getMessage(), e);
			}
			finally
			{
				try
				{
					in.close();
				}
				catch (Exception e)
				{
				}
			}
		}
	}

	/**
	 * <P>
	 * t[[Nł́AvOCIuWFNg̗LԂI点Oɂ̃\bhĂяo܂B
	 * ׂẴ\[X܂B
	 * </P>
	 * @exception PluginException ̃\bhɂăX[A炩̃G[ŏQƂ
	 */
	public void pluginRemove()
		throws PluginException
	{
		pluginContext = null;
	}

}
