package MojoWiki::Plugins::page;

use Encode;
use FindBin;

use lib join( '/', $FindBin::Bin, '..', '..' );
use MojoWiki::Constants;
use MojoWiki::PageParts;

use base qw(
    MojoWiki::Plugins::AbstractCommandPlugin
    MojoWiki::Plugins::AbstractInlinePlugin
);

sub new {
    my ( $class, $controller, @args ) = @_;

    my $self = {};

    bless( $self, $class );

    $self->{'commands'}       = [ 'enter_new_path', 'create_page' ];
    $self->{'inline_plugins'} = [ 'toc',            'newest_pages' ];
    $self->{'db'}             = $controller->app->db;

    return $self;
}

sub process_command {
    my ( $self, $controller, $command ) = @_;

    if ( $command eq 'enter_new_path' ) {
        &_enter_new_path( $self, $controller );
    }
    elsif ( $command eq 'create_page' ) {
        &_create_page( $self, $controller );
    }
    else {
        die 'unprocessable!!';
    }
}

sub _enter_new_path {
    my ( $self, $controller ) = @_;

    my $path = $controller->stash('path');

    my $common_parts       = new MojoWiki::PageParts($controller);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    $controller->render(
        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'plugin/page/enter_new_path',
        'handler'  => 'tt',
        'format'   => 'html'
    );

    $controller->rendered;
}

sub _create_page {
    my ( $self, $controller ) = @_;

    my $new_path = $controller->tx->req->params->param('new_path');
    $new_path =~ s/^\///;

    $controller->redirect_to( '/' . $new_path . '?cmd=edit' );
}

sub process_inline_plugin {
    my ( $self, $plugin_name, $path, $args_str ) = @_;

    my @args = split( /\s+/, $args_str );

    if ( $plugin_name eq 'toc' ) {
        return &_toc($self);
    }
    elsif ( $plugin_name eq 'newest_pages' ) {
        return &_newest_pages( $self, @args );
    }
    else {
        die 'unprocessable!!';
    }
}

sub _toc {
    my ($self) = @_;

    my $jquery_js_url = STATIC_URL_PREFIX . 'js/' . 'jquery.js';
    my $toc_js_url    = STATIC_URL_PREFIX . 'js/' . 'toc.js';

    my $html = <<HTML;
<br>
<script type=\"text/javascript\" src=\"$jquery_js_url\"></script>
<script type=\"text/javascript\" src=\"$toc_js_url\"></script>
<div id=\"toc\"></div>
<br>
HTML

    return Encode::decode_utf8($html);
}

sub _newest_pages {
    my ( $self, $n ) = @_;

    my @pages = $self->{'db'}->get_newest_page($n);

    my $html;
    my $current_modified_day = '';
    foreach my $page (@pages) {
        my $title = $page->{'Title'};
        if ( $title eq '' ) {
            $title = '(無題)';
        }
        my $path  = $page->{'Path'};
        my $mtime = $page->{'ModifiedTime'};

        my ( $sec, $min, $hour, $mday, $month, $year ) = localtime($mtime);
        my $modified_day
            = sprintf( "%4d-%2d-%2d", $year + 1900, $month + 1, $mday );
        if ( $modified_day ne $current_modified_day ) {
            $html .= "<b>$modified_day</b><br>\n";
            $current_modified_day = $modified_day;
        }

        $html
            .= "&nbsp;&nbsp;<a href=\""
            . URL_PREFIX
            . $path . "\">"
            . $title
            . "</a><br>\n";
    }

    return Encode::encode_utf8($html);
}

1;
