package MojoWiki::Plugins::attach;

use Digest::MD5;
use Encode;
use FindBin;

use lib join( '/', $FindBin::Bin, '..', '..' );
use MojoWiki::Constants;
use MojoWiki::PageParts;

use base qw(
    MojoWiki::Plugins::AbstractCommandPlugin
    MojoWiki::Plugins::AbstractInlinePlugin
);

use constant 'DATA_DIR' => '__PREFIX__/var/data';

sub new {
    my ( $class, $controller, @args ) = @_;

    my $self = {};

    bless( $self, $class );

    $self->{'commands'}
        = [ 'select_attachment', 'attach', 'download', 'delete' ];
    $self->{'inline_plugins'} = [ 'attachments', 'file' ];
    $self->{'db'} = $controller->app->db;

    return $self;
}

sub process_command {
    my ( $self, $controller, $command ) = @_;

    if ( $command eq 'select_attachment' ) {
        &_select_attachment( $self, $controller );
    }
    elsif ( $command eq 'attach' ) {
        &_attach( $self, $controller );
    }
    elsif ( $command eq 'download' ) {
        &_download( $self, $controller );
    }
    elsif ( $command eq 'delete' ) {
        &_delete( $self, $controller );
    }
    else {
        die 'unprocessable!!';
    }
}

sub _select_attachment {
    my ( $self, $controller ) = @_;

    my $path = $controller->stash('path');

    my $page = $self->{'db'}->get_page($path);
    if ( !defined($page) ) {

 # TODO
 # ページがまだ存在しない場合はエラーメッセージを表示
    }

    my $common_parts       = new MojoWiki::PageParts($controller);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    $controller->render(
        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'plugin/attach/select_attachment',
        'handler'  => 'tt',
        'format'   => 'html'
    );

    $controller->rendered;
}

sub _attach {
    my ( $self, $controller ) = @_;

    my $path = $controller->stash('path');

    my $page = $self->{'db'}->get_page($path);
    if ( !defined($page) ) {
        $page->{'Title'}    = '/' . $path;
        $page->{'Revision'} = 0;

 # TODO
 # ページがまだ存在しない場合はエラーメッセージを表示
    }

    # Mojo::Upload
    my $file = $controller->tx->req->upload('attachment');

    if ( !defined($file) ) {
        $controller->error('ファイルが指定されていません');

        return;
    }

    my $filename = Encode::encode_utf8( $file->filename );
    my $hash     = Digest::MD5::md5_hex( $path . $filename );
    $file->move_to( join( '/', DATA_DIR, $hash ) );

    # DBに元ファイル名のリストを格納
    my @filenames
        = $controller->app->db->plugin_get( 'attach', "$path-filenames" );
    if ( grep( /^$filename$/, @filenames ) ) {

# 重複しているファイルがあったら、ファイル名の末尾に番号をつける
        my $count = 1;
        my $filename = &_add_number_to_filename( $filename, $count );
        while ( grep( /^$filename$/, @filenames ) ) {
            $count++;
            $filename = &_add_number_to_filename( $filename, $count );
        }
    }
    push( @filenames, $filename );
    $controller->app->db->plugin_set( 'attach', "$path-filenames",
        @filenames );

    # DBに 元ファイル名 -> ハッシュ の情報を格納
    $controller->app->db->plugin_set( 'attach', "$path-hash-$filename",
        $hash );

    # DBに ハッシュ -> 元ファイル名 の情報を格納
    $controller->app->db->plugin_set( 'attach', "$path-name-$hash",
        $filename );

    my $common_parts       = new MojoWiki::PageParts($controller);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    $controller->render(
        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'plugin/attach/attach',
        'handler'  => 'tt',
        'format'   => 'html'
    );

    $controller->rendered;
}

sub _add_number_to_filename {
    my ( $filename, $n ) = @_;

    if ( $filename =~ s/(\.[^\.]*)$/_$n$1/ ) {
    }
    else {
        $filename =~ s/$/_$n/;
    }

    return $filename;
}

sub _download {
    my ( $self, $controller ) = @_;

    my $path = $controller->stash('path');
    my $hash = $controller->tx->req->params->param('hash');

    my ($filename)
        = $self->{'db'}->plugin_get( 'attach', "$path-name-$hash" );

# TODO
# IEはこれで文字化けなし、他のブラウザはとりあえず放置
    $filename = Encode::encode( 'shift-jis', Encode::decode_utf8($filename) );

    $controller->tx->res->headers->header(
        'Content-Disposition' => "attachment; filename=\"$filename\"" );

    my $file_path = join( '/', DATA_DIR, $hash );
    my $relative_path = &_get_relative_path( STATIC_DIR, $file_path );

    $controller->render_static($relative_path);

    $controller->rendered;
}

sub _delete {
    my ( $self, $controller ) = @_;

    my $path = $controller->stash('path');
    my $hash = $controller->tx->req->params->param('hash');

    my $page = $self->{'db'}->get_page($path);
    if ( !defined($page) ) {
        $page->{'Title'}    = '/' . $path;
        $page->{'Revision'} = 0;

       # TODO
       # ページが存在しない場合はエラーメッセージを表示
    }

    my $file_path = join( '/', DATA_DIR, $hash );
    unlink($file_path);

    # DBのファイル名リストから削除
    my ($target_filename)
        = $controller->app->db->plugin_get( 'attach', "$path-name-$hash" );
    my @filenames
        = $controller->app->db->plugin_get( 'attach', "$path-filenames" );
    my @new_filenames;
    foreach my $filename (@filenames) {
        if ( $filename eq $target_filename ) {
            next;
        }
        push( @new_filenames, $filename );
    }
    $controller->app->db->plugin_set( 'attach', "$path-filenames",
        @new_filenames );

    # DBから 元ファイル名 -> ハッシュ の情報を削除
    $controller->app->db->plugin_set( 'attach',
        "$path-hash-$target_filename" );

    # DBから ハッシュ -> 元ファイル名 の情報を削除
    $controller->app->db->plugin_set( 'attach', "$path-name-$hash" );

    my $common_parts       = new MojoWiki::PageParts($controller);
    my $header_contents    = $common_parts->get_header();
    my $navigator_contents = $common_parts->get_navigator();
    my $footer_contents    = $common_parts->get_footer();

    $controller->render(
        'header_contents'    => Encode::encode_utf8($header_contents),
        'navigator_contents' => Encode::encode_utf8($navigator_contents),
        'footer_contents'    => Encode::encode_utf8($footer_contents),

        'template' => 'plugin/attach/delete',
        'handler'  => 'tt',
        'format'   => 'html'
    );

    $controller->rendered;
}

sub _get_relative_path {
    my ( $from_path, $to_path ) = @_;

    my @from_dirs = split( /\//, $from_path );
    my @to_dirs   = split( /\//, $to_path );

    my $n = scalar(@from_dirs);
    for ( my $i = 0; $i < $n; $i++ ) {
        if ( !defined( $from_dirs[0] ) || !defined( $to_dirs[0] ) ) {
            next;
        }
        if ( $from_dirs[0] eq $to_dirs[0] ) {
            shift(@from_dirs);
            shift(@to_dirs);
        }
    }

    for ( my $i = 0; $i < scalar(@from_dirs); $i++ ) {
        unshift( @to_dirs, '..' );
    }

    return join( '/', @to_dirs );
}

sub process_inline_plugin {
    my ( $self, $plugin_name, $path, $args_str ) = @_;

    my @args = split( /\s+/, $args_str );

    if ( $plugin_name eq 'attachments' ) {
        return &_attachments( $self, $path );
    }
    elsif ( $plugin_name eq 'file' ) {
        return &_file( $self, $path, @args );
    }
    else {
        die 'unprocessable!!';
    }
}

sub _attachments {
    my ( $self, $path ) = @_;

    my @filenames = $self->{'db'}->plugin_get( 'attach', "$path-filenames" );
    my @links;
    foreach my $filename (@filenames) {
        my ($hash)
            = $self->{'db'}->plugin_get( 'attach', "$path-hash-$filename" );
        my $link
            = '<a href="' 
            . $path
            . '?cmd=download&hash='
            . $hash . '">'
            . $filename . '</a>['
            . '<a href="'
            . $path
            . '?cmd=delete&hash='
            . $hash
            . '">削除</a>]';
        push( @links, $link );
    }

    return join( ',', @links );
}

sub _file {
    my ( $self, $path, $filename, $width, $height ) = @_;

    my @filenames = $self->{'db'}->plugin_get( 'attach', "$path-filenames" );

    if ( !grep( /^$filename$/, @filenames ) ) {
        return "file \'$filename\' not found.";
    }

    my $size_str = '';
    if ( defined($width) ) {
        $size_str = $size_str . ' width=' . $width;
    }
    if ( defined($height) ) {
        $size_str = $size_str . ' height=' . $height;
    }

    my ($hash)
        = $self->{'db'}->plugin_get( 'attach', "$path-hash-$filename" );

    my $link;
    if ( $filename =~ /\.(png)|(jpg)|(bmp)$/ ) {
        $link
            = '<img src="' 
            . $path
            . '?cmd=download&hash='
            . $hash . '"'
            . $size_str
            . ' style="max-width: 80%">';
    }
    else {
        $link
            = '<a href="' 
            . $path
            . '?cmd=download&hash='
            . $hash . '">'
            . $filename . '</a>';
    }

    return $link;
}

sub delete_page {
    my ( $self, $path ) = @_;

    my @filenames = $self->{'db'}->plugin_get( 'attach', "$path-filenames" );
    $self->{'db'}->plugin_set( 'attach', "$path-filenames" );

    foreach my $filename (@filenames) {
        my ($hash)
            = $self->{'db'}->plugin_get( 'attach', "$path-hash-$filename" );

        $self->{'db'}->plugin_set( 'attach', "$path-hash-$filename" );
        $self->{'db'}->plugin_set( 'attach', "$path-name-$hash" );

        unlink( join( '/', DATA_DIR, $hash ) );
    }
}

1;
